/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.IO;
using GME.CSharp;
using GME.MGA;
using GME;
using GME.MGA.Core;
using System.Windows.Forms;
using System.Linq;
using Excel = Microsoft.Office.Interop.Excel;
using GME.MGA.Meta;

namespace CyPhyMdaoAddOn
{
	[Guid(ComponentConfig.guid),
	ProgId(ComponentConfig.progID),
	ClassInterface(ClassInterfaceType.AutoDual)]
	[ComVisible(true)]
	public class CyPhyMdaoAddOnAddon : IMgaComponentEx, IGMEVersionInfo, IMgaEventSink
	{

		private MgaAddOn addon;
		private bool componentEnabled = true;

		// Event handlers for addons
		#region MgaEventSink members
		public void GlobalEvent(globalevent_enum @event)
		{
			if (@event == globalevent_enum.GLOBALEVENT_CLOSE_PROJECT)
			{
				Marshal.FinalReleaseComObject(addon);
				addon = null;
			}
			if (!componentEnabled)
			{
				return;
			}

			//TODO: Handle global events
			//MessageBox.Show(@event.ToString());
		}

		/// <summary>
		/// Called when an FCO or folder changes
		/// </summary>
		/// <param name="subject">the object the event(s) happened to</param>
		/// <param name="eventMask">objectevent_enum values ORed together</param>
		/// <param name="param">extra information provided for cetertain event types</param>
		public void ObjectEvent(MgaObject subject, uint eventMask, object param)
		{
			if (!componentEnabled)
			{
				return;
			}

			// TODO: Handle object events (OR eventMask with the members of objectevent_enum)
			// Warning: Only those events are received that you have subscribed for by setting ComponentConfig.eventMask
			uint uOBJEVENT_ATTR = 0;
			uint uOBJEVENT_CONNECTED = 0;
			uint uOBJEVENT_CREATED = 0;

			unchecked { uOBJEVENT_ATTR = (uint)objectevent_enum.OBJEVENT_ATTR; }
			unchecked { uOBJEVENT_CONNECTED = (uint)objectevent_enum.OBJEVENT_CONNECTED; }
			unchecked { uOBJEVENT_CREATED = (uint)objectevent_enum.OBJEVENT_CREATED; }

			if ((eventMask & uOBJEVENT_CONNECTED) != 0)
			{

			}
			else if ((eventMask & uOBJEVENT_CREATED) != 0)
			{
				if (subject.MetaBase.Name == "TestBenchRef")
				{
					// set the port label lenght 0
					(subject as MgaFCO).RegistryValue["portLabelLength"] = "0";
					if (subject as MgaReference != null)
					{
						UpdateColor((subject as MgaReference).Referred);
					}
				}
				else if (subject.MetaBase.Name == "Optimizer")
				{
					(subject as MgaFCO).RegistryValue["portLabelLength"] = "0";
					UpdateColor(subject as MgaFCO);
				}
				else if (subject.MetaBase.Name == "VariableSweep")
				{
					(subject as MgaFCO).RegistryValue["dstStyle"] = "arrow";
					(subject as MgaFCO).RegistryValue["color"] = "0xff80c0";
				}
				else if (subject.MetaBase.Name == "ObjectiveMapping")
				{
					(subject as MgaFCO).RegistryValue["dstStyle"] = "arrow";
					(subject as MgaFCO).RegistryValue["color"] = "0x7cadde";
				}
			}
			else if ((eventMask & uOBJEVENT_ATTR) != 0)
			{
				if (subject.MetaBase.Name == "ValueFlow")
				{
					// set the src custom formula's name to the specified name
					// in the ValueFlow attribute field
					MgaConnection ValueFlow = subject as MgaConnection;
					foreach (MgaConnPoint cp in ValueFlow.ConnPoints)
					{
						if (cp.ConnRole == "src")
						{
							if (cp.Target.Meta.Name == "CustomFormula")
							{
								foreach (MgaAttribute attr in ValueFlow.Attributes)
								{
									if (attr.Meta.Name == "FormulaVariableName")
									{
										if (string.IsNullOrEmpty(attr.StringValue) == false)
										{
											cp.Target.Name = attr.StringValue;
											break;
										}
									}
								}
							}
							break;
						}
					}
				}
				else if (subject.MetaBase.Name == "TestBench")
				{
					// Excel import helper
					MgaModel testBench = subject as MgaModel;

					// call color helper
					UpdateColor(testBench as MgaFCO);

					if (testBench != null)
					{
						try
						{
							string type = testBench.StrAttrByName["Type"];
							if (type == "Excel" &&
								testBench.ChildObjects.Count == 0)
							{
								// excel and empty test bench
								using (OpenFileDialog fd =
									new OpenFileDialog()
									{
										DefaultExt = "xls",
										Filter = "Excel (*.xls)|*.xls|Excel 2010 (*.xlsx)|*.xlsx|All files (*.*)|*.*",
										Multiselect = false,
										InitialDirectory = Path.GetDirectoryName(subject.Project.ProjectConnStr.Substring("MGA=".Length)),
									})
								{

									DialogResult result = fd.ShowDialog();
									if (result == DialogResult.OK)
									{
										if (string.IsNullOrEmpty(fd.FileName))
										{
											// TODO: error message
											return;
										}
										testBench.StrAttrByName["SpreadsheetLoc"] = fd.FileName;
										ExcelImportForm form = new ExcelImportForm();

										Excel.Application xlApp = new Excel.Application();
										Excel.Workbook wb = xlApp.Workbooks.Open(fd.FileName);
										Excel.Names names = wb.Names;
										List<string> RangeNames = new List<string>();
										foreach (Excel.Name name in names)
										{
											// get named cells
											RangeNames.Add(name.Name);
										}


										form.lbNamedCells.Items.Clear();
										RangeNames.ForEach(x => form.lbNamedCells.Items.Add(x));

										result = form.ShowDialog();
										if (result == DialogResult.OK)
										{
											MgaMetaRole role = null;
											MgaFCO fco = null;
											int StartX = 20;
											int StartY = 20;
											int PaddingX = 300;
											int PaddingY = 100;

											int i = 0;

											role = (testBench.Meta as MgaMetaModel).RoleByName["Parameter"];
											// create parameters
											foreach (string item in form.lbParameters.Items)
											{
												fco = testBench.CreateChildObject(role);
												fco.Name = item;
												fco.StrAttrByName["Value"] = xlApp.Range[names.Item(Index: item).RefersToLocal].Value.ToString();

												foreach (IMgaPart part in fco.Parts)
												{
													part.SetGmeAttrs(null, StartX, StartY + PaddingY * i);
												}
												i++;
											}

											i = 0;
											role = (testBench.Meta as MgaMetaModel).RoleByName["Metric"];
											// create metrics
											foreach (string item in form.lbMetrics.Items)
											{
												fco = testBench.CreateChildObject(role);
												fco.Name = item;
												// we don't need default value for metrics

												foreach (IMgaPart part in fco.Parts)
												{
													part.SetGmeAttrs(null, StartX + PaddingX, StartY + PaddingY * i);
												}
												i++;
											}

										}
										// TODO: need to be placed in the finally block!
										wb.Close(SaveChanges: false);
										xlApp.Quit();
									}

								}

							}
						}
						catch (Exception ex)
						{
                            System.Diagnostics.Debug.WriteLine(ex);
						}
					}
				}
			}


			//MessageBox.Show(eventMask.ToString());

		}

		private void UpdateColor(MgaFCO subject)
		{
			// default values
			string shadowThickness = "12";
			string itemShadowCast = "true";
			string shadowColor = "0x666666";
			string color = "0xc0c0c0";
			string borderColor = "0x000000";

			if (subject != null)
			{
				if (subject.Meta.Name == "TestBench")
				{
					string Type = subject.StrAttrByName["Type"];
					
					if (Type == "Excel")
					{
						color = "0x98fb98";
						borderColor = "0x005500";
					}
					else if (Type == "DynSim")
					{
						color = "0x7d9ec0";
						borderColor = "0x00008b";
					}
					else if (Type == "SimpleCalc")
					{
						color = "0xffad5b";
						borderColor = "0xa52a00";
					}
				}
				else if (subject.Meta.Name == "Optimizer")
				{
					color = "0xff6820";
					borderColor = "0x8b0000";
				}

				subject.RegistryValue["color"] = color;
				subject.RegistryValue["borderColor"] = borderColor;
				subject.RegistryValue["shadowThickness"] = shadowThickness;
				subject.RegistryValue["itemShadowCast"] = itemShadowCast;
				subject.RegistryValue["shadowColor"] = shadowColor;
			}
		}

		#endregion

		#region IMgaComponentEx Members

		public void Initialize(MgaProject p)
		{
			// Creating addon
			p.CreateAddOn(this, out addon);
			// Setting event mask (see ComponentConfig.eventMask)
			unchecked
			{
				addon.EventMask = (uint)ComponentConfig.eventMask;
			}
		}

		public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
		{
			throw new NotImplementedException(); // Not called by addon
		}


		#region Component Information
		public string ComponentName
		{
			get { return GetType().Name; }
		}

		public string ComponentProgID
		{
			get
			{
				return ComponentConfig.progID;
			}
		}

		public componenttype_enum ComponentType
		{
			get { return ComponentConfig.componentType; }
		}
		public string Paradigm
		{
			get { return ComponentConfig.paradigmName; }
		}
		#endregion

		#region Enabling
		bool enabled = true;
		public void Enable(bool newval)
		{
			enabled = newval;
		}
		#endregion

		#region Interactive Mode
		protected bool interactiveMode = true;
		public bool InteractiveMode
		{
			get
			{
				return interactiveMode;
			}
			set
			{
				interactiveMode = value;
			}
		}
		#endregion

		#region Custom Parameters
		SortedDictionary<string, object> componentParameters = null;

		public object get_ComponentParameter(string Name)
		{
			if (Name == "type")
				return "csharp";

			if (Name == "path")
				return GetType().Assembly.Location;

			if (Name == "fullname")
				return GetType().FullName;

			object value;
			if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
			{
				return value;
			}

			return null;
		}

		public void set_ComponentParameter(string Name, object pVal)
		{
			if (componentParameters == null)
			{
				componentParameters = new SortedDictionary<string, object>();
			}

			componentParameters[Name] = pVal;
		}
		#endregion

		#region Unused Methods
		// Old interface, it is never called for MgaComponentEx interfaces
		public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
		{
			throw new NotImplementedException();
		}

		// Not used by GME
		public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
		{
			throw new NotImplementedException();
		}

		#endregion

		#endregion

		#region IMgaVersionInfo Members

		public GMEInterfaceVersion_enum version
		{
			get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
		}

		#endregion

		#region Registration Helpers

		[ComRegisterFunctionAttribute]
		public static void GMERegister(Type t)
		{
			Registrar.RegisterComponentsInGMERegistry();
		}

		[ComUnregisterFunctionAttribute]
		public static void GMEUnRegister(Type t)
		{
			Registrar.UnregisterComponentsInGMERegistry();
		}

		#endregion
	}
}
