/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CyPhyMetricEvaluator.h"
#include "UmlExt.h"


CyPhyMetricEvaluator::CyPhyMetricEvaluator(CyPhyML::TestBench tb)
	:testBench(tb)
{

}

void CyPhyMetricEvaluator::evaluate()
{
	// test bench is already exists
	set<CyPhyML::Metric> metrics = testBench.Metric_kind_children();
	for(set<CyPhyML::Metric>::iterator i = metrics.begin();
		i != metrics.end();
		++i)
	{
		//evaluate(CyPhyML::Metric(*i));
		std::string val = i->Value();
		if(val.empty())
			val = "n/a";
		metricValueMap[i->name()] = val;
		processedMetrics.insert(*i);
	}
	// in case of caref is null
	cfgName = testBench.name();

	CyPhyML::TopLevelSystemUnderTest caref = testBench.TopLevelSystemUnderTest_child();
	if(caref == Udm::null) return;

	CyPhyML::DesignEntity de = caref.ref();
	if(!Uml::IsDerivedFrom(de.type(), CyPhyML::ComponentAssembly::meta))
		return;

	CyPhyML::ComponentAssembly ca = CyPhyML::ComponentAssembly::Cast(de);
	cfgUniqueId = ca.ConfigurationUniqueID();
	
	cfgName = ca.name();
	cfgName = cfgName.substr(4);


}

map<std::string, std::string> CyPhyMetricEvaluator::getMetricMap()
{
	return metricValueMap;
}

string CyPhyMetricEvaluator::getCfgUniqueId()
{
	return cfgUniqueId;
}

std::string CyPhyMetricEvaluator::getCfgName()
{
	return cfgName;
}

void CyPhyMetricEvaluator::evaluate(CyPhyML::Metric &metric)
{
	if(processedMetrics.find(metric)!=processedMetrics.end())
		return;

	std::string metric_name = metric.name();
	set<CyPhyML::ValueFlow> vfs = metric.srcValueFlow();
	if(vfs.empty()) return;
	if(vfs.size()>1)
		throw udm_exception(metric_name+" has more than one source ValueFlow.");

	CyPhyML::ValueFlow vf = *(vfs.begin());
	CyPhyML::ValueFlowTarget src_vfend = vf.srcValueFlow_end();

	const Uml::Class &vfType = src_vfend.type();
	if(vfType == CyPhyML::Property::meta)
		metric.Value() = CyPhyML::Property::Cast(src_vfend).Value();
	else if(vfType == CyPhyML::Parameter::meta)
		metric.Value() = CyPhyML::Parameter::Cast(src_vfend).Value();
	else if(vfType == CyPhyML::CADParameter::meta)
		metric.Value() = CyPhyML::CADParameter::Cast(src_vfend).Value();
	else if(vfType == CyPhyML::Metric::meta)
	{
		CyPhyML::Metric src_metric = CyPhyML::Metric::Cast(src_vfend);
		evaluate(src_metric);
		metric.Value() = src_metric.Value();
	}
	else if(vfType == CyPhyML::SimpleFormula::meta)
	{
		metric.Value() = to_string(evaluate(CyPhyML::SimpleFormula::Cast(src_vfend)));
	}
	else if(vfType == CyPhyML::CustomFormula::meta)
	{
		metric.Value() = to_string(evaluate(CyPhyML::CustomFormula::Cast(src_vfend)));
	}

	std::string val = metric.Value();
	if(val.empty())
		val = "n/a";
	metricValueMap[metric_name] = val;
	processedMetrics.insert(metric);
}

long double CyPhyMetricEvaluator::evaluate(CyPhyML::SimpleFormula &sformula)
{
	double ret = 0;
	
	vector<double> parameters;
	set<CyPhyML::ValueFlow> src_vfs = sformula.srcValueFlow();
	for(set<CyPhyML::ValueFlow>::iterator i=src_vfs.begin();i!=src_vfs.end();++i)
	{
		std::string  val;
		CyPhyML::ValueFlowTarget src_vfend = (*i).srcValueFlow_end();
		const Uml::Class &vfType = src_vfend.type();
		if(vfType == CyPhyML::Property::meta)
			val = CyPhyML::Property::Cast(src_vfend).Value();
		else if(vfType == CyPhyML::Parameter::meta)
			val = CyPhyML::Parameter::Cast(src_vfend).Value();
		else if(vfType == CyPhyML::CADParameter::meta)
			val = CyPhyML::CADParameter::Cast(src_vfend).Value();
		else if(vfType == CyPhyML::Metric::meta)
		{
			CyPhyML::Metric src_metric = CyPhyML::Metric::Cast(src_vfend);
			evaluate(src_metric);
			val = src_metric.Value();
		}
		else
			throw udm_exception((std::string)sformula.name()+" has wrong incoming connections.");

		if(val.empty())
			parameters.push_back(0);
		else
			parameters.push_back(atof(val.c_str()));
	}

	std::string operation = sformula.Method();
	try 
	{
		mu::Parser parser;

		if (operation == "Addition")
		{	
			std::string expression = "sum(" + createExpression(parameters, parser) + ")";
			parser.SetExpr(expression);
			ret = parser.Eval();		
		}
		else if (operation == "Multiplication")
		{
			//result = EvaluateMultiplication(parameters);
			parser.SetExpr(createExpression(parameters, parser, "*"));
			ret = parser.Eval(); 
		}
		else if (operation == "ArithmeticMean")
		{
			std::string expression = "avg(" + createExpression(parameters, parser) + ")";
			parser.SetExpr(expression);
			ret = parser.Eval();	
		}
		else if (operation == "GeometricMean")
			ret = getGeometriMeanValue(parameters);
		else if (operation == "Maximum")
		{
			std::string expression = "max(" + createExpression(parameters, parser) + ")";
			parser.SetExpr(expression);
			ret = parser.Eval();	
		}
		else if (operation == "Minimum")
		{
			std::string expression = "min(" + createExpression(parameters, parser) + ")";
			parser.SetExpr(expression);
			ret = parser.Eval();	
		}
	}
	catch (mu::Parser::exception_type &e)
	{
		throw udm_exception("muParser Exception: " + e.GetMsg());
	}
	catch (std::exception &e)
	{
		throw udm_exception("Exception: " + (std::string)e.what());
	}

	return ret;
}

long double CyPhyMetricEvaluator::evaluate(CyPhyML::CustomFormula &cformula)
{
	double ret = 0;

	std::string fname = cformula.name();
	map<std::string, double> parameters;
	set<CyPhyML::ValueFlow> src_vfs = cformula.srcValueFlow();
	for(set<CyPhyML::ValueFlow>::iterator i=src_vfs.begin();i!=src_vfs.end();++i)
	{
		std::string  val;
		CyPhyML::ValueFlowTarget src_vfend = (*i).srcValueFlow_end();
		const Uml::Class &vfType = src_vfend.type();
		if(vfType == CyPhyML::Property::meta)
			val = CyPhyML::Property::Cast(src_vfend).Value();
		else if(vfType == CyPhyML::Parameter::meta)
			val = CyPhyML::Parameter::Cast(src_vfend).Value();
		else if(vfType == CyPhyML::CADParameter::meta)
			val = CyPhyML::CADParameter::Cast(src_vfend).Value();
		else if(vfType == CyPhyML::Metric::meta)
		{
			CyPhyML::Metric src_metric = CyPhyML::Metric::Cast(src_vfend);
			evaluate(src_metric);
			val = src_metric.Value();
		}
		else
			throw udm_exception((std::string)cformula.name()+" has wrong incoming connections.");

		if(val.empty())
			parameters[(std::string)src_vfend.name()] = 0;
		else
			parameters[(std::string)src_vfend.name()] = atof(val.c_str());
	}

	std::string expression = cformula.Expression();
	try
	{
		mu::Parser parser;
		parser.ResetLocale();

		for(map<std::string, double>::iterator i=parameters.begin(); i!=parameters.end(); i++)
			parser.DefineVar((*i).first, &((*i).second));

		parser.SetExpr(CheckExpression(expression));
		ret = parser.Eval();
	}
	catch (mu::Parser::exception_type &e)
	{
		throw udm_exception("muParser Exception: " + e.GetMsg());
	}
	catch (std::exception &e)
	{
		throw udm_exception("Exception: " + (std::string)e.what());
	}
	catch (...)
	{
		throw udm_exception("Exception! Not sure what though!");
	}

	return ret;
}

std::string CyPhyMetricEvaluator::createExpression(vector<double> &parameters, mu::Parser &parser, std::string delimiter)
{
	std::string expression;
	for(vector<double>::iterator i=parameters.begin();i!=parameters.end();++i)
	{
		if (i != parameters.begin())
			expression += delimiter;

		long double val = *i;
		expression += to_string(val);  
	}

	return expression;
}

double CyPhyMetricEvaluator::getGeometriMeanValue(vector<double> &parameters)
{
	double ret = 1;
	for(vector<double>::iterator i=parameters.begin();i!=parameters.end();++i)
	{
		ret *= *i;
	}
	
	if(!parameters.empty())
	{
		double x = 1/parameters.size();
		ret = pow(ret, x);
	}
	return ret;
}

std::string CyPhyMetricEvaluator::CheckExpression(std::string expression)
{
	std::string cleanExpression;

	for (int i = 0; i < expression.size(); i++)
	{
		if (expression[i] != '\n' && expression[i] != '\t')
			cleanExpression += expression[i];
	}
	return cleanExpression;
}
