/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Core;
using System.Linq;

using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using System.Diagnostics;

namespace CyPhyPET
{
    /// <summary>
    /// This class implements the necessary COM interfaces for a GME interpreter component.
    /// </summary>
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class CyPhyPETInterpreter : IMgaComponentEx, IGMEVersionInfo
    {
        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        public enum ComponentStartMode
        {
            GME_MAIN_START = 0, 		// Not used by GME
            GME_BROWSER_START = 1,      // Right click in the GME Tree Browser window
            GME_CONTEXT_START = 2,		// Using the context menu by right clicking a model element in the GME modeling window
            GME_EMBEDDED_START = 3,		// Not used by GME
            GME_MENU_START = 16,		// Clicking on the toolbar icon, or using the main menu
            GME_BGCONTEXT_START = 18,	// Using the context menu by right clicking the background of the GME modeling window
            GME_ICON_START = 32,		// Not used by GME
            GME_SILENT_MODE = 128 		// Not used by GME, available to testers not using GME
        }

        private string logFileName;

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a tansaction.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        public void Initialize(MgaProject project)
        {
            this.componentParameters = new SortedDictionary<string, object>();
            this.componentParameters.Add("output_dir", string.Empty);
            this.componentParameters.Add("labels", string.Empty);
            this.componentParameters.Add("runCommand", "exit 0");
            this.componentParameters.Add("automation", "false");
            this.componentParameters.Add("do_config", "true");
            this.componentParameters.Add("configuration", string.Empty);
            this.componentParameters.Add("TestBench", string.Empty);
            this.componentParameters.Add("TestBenchUniqueName", string.Empty);
        }

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is avaliable. A general try-catch block catches all the exceptions
        /// coming from this function, you don't need to add it. For more information, see InvokeEx.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        /// <param name="currentobj">The model open in the active tab in GME. Its value is null if no model is open (no GME modeling windows open). </param>
        /// <param name="selectedobjs">
        /// A collection for the selected  model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the context menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the collection is empty (contains zero elements).
        /// </param>
        /// <param name="startMode">Contains information about the GUI event that initiated the invocation.</param>
        public void Main(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            ComponentStartMode startMode)
        {
            try
            {
                if (CheckTestBenchLimitations(currentobj) == false)
                {
                    return;
                }
                
                List<string> messages = new List<string>();
                // Get RootFolder
                //IMgaFolder rootFolder = project.RootFolder;
                //GMEConsole.Out.WriteLine(rootFolder.Name);
                bool hasModelicaTestBench = true;
                if (currentobj != null)
                {
                    GMEConsole.Out.WriteLine("Running {0}...", ComponentName);

                    if (currentobj.Meta.Name == "ParametricExploration")
                    {
                        CyPhy.ParametricExploration pet = ISIS.GME.Common.Utils.CreateObject<CyPhyClasses.ParametricExploration>(currentobj as MgaObject);

                        // Check that there is exactly one driver; DOE(parameter study)/PCCDriver/Optimizer.
                        bool isParameterStudy = pet.Children.ParameterStudyCollection.Count() == 1;
                        bool isOptimizer = pet.Children.OptimizerCollection.Count() == 1;
                        bool isPCC = pet.Children.PCCDriverCollection.Count() == 1;
                        int totalDrivers = pet.Children.ParameterStudyCollection.Count() 
                                            + pet.Children.OptimizerCollection.Count()
                                            + pet.Children.PCCDriverCollection.Count();

                        if (totalDrivers == 0)
                        {
                            messages.Add(
                                "In PET you need to have atleast one driver (PCCdriver/Optimizer/Parameter Study).");
                        }
                        else if (totalDrivers > 1)
                        {
                            messages.Add(
                                "In PET you can only have one driver (PCCdriver/Optimizer/Parameter Study)");
                        }

                        if (isParameterStudy)
                        {
                            var paramStudy = pet.Children.ParameterStudyCollection.FirstOrDefault();
                            foreach (var item in paramStudy.Children.DesignVariableCollection)
                            {
                                if (String.IsNullOrEmpty(item.Attributes.Range))
                                {
                                    messages.Add("DesignVariable has no range defined.");
                                }
                                else
                                {
                                    if (item.Attributes.Range.IndexOf(", ") < 0)
                                    {
                                        messages.Add
                                            ("DesignVariable's Range is not in the right format MIN, MAX.");
                                    }
                                }
                            }
                        }
                        else if (isOptimizer)
                        {
                            var optimizer = pet.Children.OptimizerCollection.FirstOrDefault();
                            foreach (var item in optimizer.Children.DesignVariableCollection)
                            {
                                if (String.IsNullOrEmpty(item.Attributes.Range))
                                {
                                    messages.Add("DesignVariable has no range defined.");
                                }
                                else
                                {
                                    if (item.Attributes.Range.IndexOf(", ") < 0)
                                    {
                                        messages.Add(
                                            "DesignVariable's Range is not in the right format MIN, MAX.");
                                    }
                                }
                            }
                        }
                        else if (isPCC)
                        {
                            var pccDriver = pet.Children.PCCDriverCollection.FirstOrDefault();
                        }

                        // Check that there is exactly on TestBench.
                        bool hasOneTestBench = pet.Children.TestBenchRefCollection.Count() == 1;

                        if (hasOneTestBench == false)
                        {
                            messages.Add("PET must have exactly one TestBechRef.");
                        }
                        else // if (hasOneTestBench)
                        {
                            var testBench = pet.Children.TestBenchRefCollection.FirstOrDefault();
                            if (testBench.Referred.TestBench == null)
                            {
                                // no testbench null ref
                                messages.Add("TestBechRef can not be null. It must point to an actual test bench.");
                            }
                            else
                            {
                                if (testBench.Referred.TestBench.Attributes.Type ==
                                    CyPhyClasses.TestBench.AttributesClass.Type_enum.Dynamics_Simulation)
                                {
                                    if (testBench.Referred.TestBench.Children.WorkflowRefCollection.Count() != 1)
                                    {
                                        // each testbench contains one workflow
                                        messages.Add("TestBechRef must contain one workflow ref. " + testBench.Name);
                                    }
                                    else
                                    {
                                        var workflow = testBench.Referred.TestBench.Children.WorkflowRefCollection.FirstOrDefault();
                                        if (workflow.Referred.Workflow == null)
                                        {
                                            messages.Add("workflow ref could not be null.");
                                        }
                                        else
                                        {
                                            if (workflow.Referred.Workflow.Children.TaskCollection.Count() != 1)
                                            {
                                                // each workflow contains one task
                                                messages.Add("workflow must contain exactly one task.");
                                            }
                                            else
                                            {
                                                var task = workflow.Referred.Workflow.Children.TaskCollection.FirstOrDefault();
                                                if (task.Attributes.COMName != "MGA.Interpreter.CyPhy2Modelica")
                                                {
                                                    messages.Add(
                                                        "Dynamics testbenches are only supported if the interpreter is CyPhy2Modelica");
                                                }
                                                else
                                                {
                                                    hasModelicaTestBench = true;
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        // verify names
                        List<string> names = new List<string>();
                        foreach (var item in pet.AllChildren.Where(
                            x => x is ISIS.GME.Common.Interfaces.Connection == false))
                        {
                            names.Add(item.Name);
                        }

                        if (names.Count != names.Distinct().Count())
                        {
                            messages.Add("Names must be unique on the PET subsystem");
                        }

                        // no error messages
                        string outputLogDir = Path.GetFullPath("log");
                        if (Directory.Exists(outputLogDir) == false)
                        {
                            Directory.CreateDirectory(outputLogDir);
                        }

                        if (messages.Count > 0)
                        {
                            using (StreamWriter writer = new StreamWriter(Path.Combine(outputLogDir, "log.txt")))
                            {
                                foreach (var item in messages)
                                {
                                    GMEConsole.Error.WriteLine(item);
                                    writer.WriteLine(item);
                                }
                            }
                        }
                        else
                        {
                            if (File.Exists(Path.Combine(outputLogDir, "log.txt")))
                            {
                                File.Delete(Path.Combine(outputLogDir, "log.txt"));
                            }


                            //string outputDirectory = Path.GetFullPath(Path.Combine("results", pet.Name));
                            string outputDirectory = (string)this.componentParameters["output_dir"]; 
                                //Path.GetFullPath(Path.Combine("results", pet.Name));
                            
                            if (Directory.Exists(outputDirectory) == false)
                            {
                                Directory.CreateDirectory(outputDirectory);
                            }

                            // generate driver + assemblies
                            PET p = new PET(pet, outputDirectory, lib_package_folder, lib_package_name)
                            {
                                HasModelicaTestBench = hasModelicaTestBench,
                            };
                            
                            // Forward info about PET-type to PET-class
                            PET.TypeOfPET PETType = new PET.TypeOfPET();
                            if (isPCC)
                            {
                                PETType = PET.TypeOfPET.PCC;
                            }
                            else if (isOptimizer)
                            {
                                PETType = PET.TypeOfPET.Optimizer;
                            }
                            else if (isParameterStudy)
                            {
                                PETType = PET.TypeOfPET.ParameterStudy;
                            }

                            p.GenerateCode(PETType);

                            using (StreamWriter writer = new StreamWriter(Path.Combine(outputDirectory, "save_results.py")))
                            {
                                writer.WriteLine(CyPhyPET.Properties.Resources.save_results);
                            }
                            //p.Manifest.ResourceFiles.Add(Path.Combine(p.RelativeSubDirectory, "save_results.py"));

                            using (StreamWriter writer = new StreamWriter(Path.Combine(outputDirectory, "mdao_index.html")))
                            {
                                writer.WriteLine(CyPhyPET.Properties.Resources.index);
                            }
                            //p.Manifest.ResourceFiles.Add(Path.Combine(p.RelativeSubDirectory, "mdao_index.html"));

                            this.componentParameters["runCommand"] = p.RunCommand;

                            GMEConsole.Info.WriteLine(
                                "Generated files are located here: <a href=\"file:///{0}\" target=\"_blank\">{0}</a>.",
                                Path.GetFullPath(outputDirectory));
                        }
                    }
                    else
                    {
                        GMEConsole.Error.WriteLine(
                            "Please start this interpreter from a ParametricExploration object.");
                    }

                    GMEConsole.Info.WriteLine("Done");
                }
                else if (currentobj == null)
                {
                    GMEConsole.Error.WriteLine(
                        "Please start this interpreter from a ParametricExploration object.");

                }
            }
            catch (Exception ex)
            {
                string logDir = Path.GetFullPath("log");
                if (Directory.Exists(logDir) == false)
                {
                    Directory.CreateDirectory(logDir);
                }
                string log_file = Path.Combine(logDir, ComponentName + ".log");

                using (StreamWriter wr = new StreamWriter(log_file))
                {
                    wr.WriteLine(DateTime.Now);
                    wr.WriteLine("==================");
                    wr.WriteLine(ex);
                }
                GMEConsole.Error.WriteLine(
                    "Log file is here: <a href=\"file:///{1}\" target=\"_blank\">{1}</a>. {0}",
                    ComponentName + ".log",
                    Path.GetDirectoryName(log_file));
            }
        }

        #region IMgaComponentEx Members

        MgaGateway MgaGateway { get; set; }
        GMEConsole GMEConsole { get; set; }

        public static string META_PATH { get; set; }
        /*public static string META_PATH_PYTHON_ACTIVATE
        {
            get
            {
                return Path.Combine(META_PATH, "bin", "Python27", "Scripts", "activate.bat");
            }
        }

        public static string META_PATH_PCC_CORE
        {
            get
            {
                return Path.Combine(META_PATH, "bin", "PCC");
            }
        }*/


        private bool CheckTestBenchLimitations(MgaFCO currentobj)
        {
            List<string> unsupportedKinds = new List<string>() {
            };

            if (currentobj.Meta.Name == "ParametricExploration")
            {
                if (currentobj.ChildObjects.Cast<MgaFCO>().Any(x => unsupportedKinds.Contains(x.Meta.Name)))
                {
                    GMEConsole.Error.WriteLine(
                        "Unsupported elements are in the PET such as Optimizer and ParameterStudy.");
                    return false;
                }

                if (currentobj.ChildObjects.Cast<MgaFCO>().Where(x => x.Meta.Name == "TestBenchRef").Count() > 1)
                {
                    GMEConsole.Error.WriteLine(
                        "Max one TestBenchRef can be used");
                    return false;
                }

                var tbref = currentobj.
                    ChildObjects.
                    OfType<MgaReference>().
                    FirstOrDefault(x => x.Meta.Name == "TestBenchRef");

                if (tbref != null &&
                    tbref.Referred == null)
                {
                    GMEConsole.Error.WriteLine(
                        "Test bench ref cannot be null");
                    return false;
                }
            }
            return true;
        }

        List<string> lib_package_folder;
        List<string> lib_package_name;

        public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            if (!enabled)
            {
                return;
            }
            string currentWorkDir = System.IO.Directory.GetCurrentDirectory();
            try
            {
                System.IO.Directory.SetCurrentDirectory(Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length)));
                string keyName = @"HKEY_LOCAL_MACHINE\Software\META";
                string value = @"META_PATH";

                // logfile in the project directory
                this.logFileName = Path.Combine(
                    Environment.CurrentDirectory,
                    "log",
                    this.ComponentName + "." + System.Diagnostics.Process.GetCurrentProcess().Id + ".log");

                META.Logger.AddFileListener(this.logFileName, this.ComponentName);

                Trace.TraceInformation("InvokeEx called");

                META_PATH = (string)Microsoft.Win32.Registry.GetValue(
                    keyName,
                    value,
                    "ERROR: " + keyName + value + " does not exist!");

                GMEConsole = GMEConsole.CreateFromProject(project);
                PET.GMEConsole = GMEConsole;
                MgaGateway = new MgaGateway(project);
                project.CreateTerritoryWithoutSink(out MgaGateway.territory);

                MgaGateway.PerformInTransaction(delegate
                {
                    var tbrefs = currentobj.
                        ChildObjects.
                        OfType<IMgaReference>().
                        Where(x => x.Meta.Name == "TestBenchRef" && x.Referred != null);

                    // ASSUMPTION: ONE Active test bench ref only, which is modelica
                    var tbref = tbrefs.FirstOrDefault();
                    // Set output dir
                    CyPhy.ParametricExploration pet = ISIS.GME.Common.Utils.CreateObject<CyPhyClasses.ParametricExploration>(currentobj as MgaObject);

                    if (string.IsNullOrEmpty(this.componentParameters["output_dir"] as string))
                    {
                        // set default output directory
                        this.componentParameters["output_dir"] = Path.GetFullPath(Path.Combine("results", pet.Name));
                    }

                    if (tbref != null)
                    {
                        // CyPhy2Modelica should only be called if TestBench is the following; 
                        // Dynamic Simulation + has a workflow + workflow has one task that is CyPhy2Modelica.

                        bool callCyPhy2Modelica = false;

                        var testBench = pet.Children.TestBenchRefCollection.FirstOrDefault().Referred.TestBench;
                        bool isDynSimTB = testBench.Attributes.Type ==
                            CyPhyClasses.TestBench.AttributesClass.Type_enum.Dynamics_Simulation;
                        bool hasWorkFlow = testBench.Children.WorkflowRefCollection.Count() == 1;

                        if (isDynSimTB && hasWorkFlow)
                        {
                            var workFlow = testBench.Children.WorkflowRefCollection.FirstOrDefault();
                            if (workFlow != null)
                            {
                                bool hasTask = workFlow.Referred.Workflow.Children.TaskCollection.Count() == 1;
                                if (hasTask)
                                {
                                    if (workFlow.Referred.Workflow.Children.TaskCollection.FirstOrDefault().
                                        Attributes.COMName == "MGA.Interpreter.CyPhy2Modelica")
                                    {
                                        callCyPhy2Modelica = true;
                                    }
                                }
                            }
                        }

                        if (callCyPhy2Modelica)
                        {
                            // call CyPhy2Modelica interpreter
                            Type t = Type.GetTypeFromProgID("MGA.Interpreter.CyPhy2Modelica");
                            IMgaComponentEx cyPhy2Modelica = Activator.CreateInstance(t) as IMgaComponentEx;

                            // dialog box from Modelica interpreter
                            cyPhy2Modelica.Initialize(project);

                            cyPhy2Modelica.ComponentParameter["output_dir"] = Path.Combine((string)this.componentParameters["output_dir"], testBench.Name);
                            cyPhy2Modelica.ComponentParameter["automation"] = this.componentParameters["automation"];
                            cyPhy2Modelica.ComponentParameter["do_config"] = this.componentParameters["do_config"];
                            cyPhy2Modelica.ComponentParameter["configuration"] = this.componentParameters["configuration"];
                            cyPhy2Modelica.ComponentParameter["TestBench"] = this.componentParameters["TestBench"];
                            cyPhy2Modelica.ComponentParameter["TestBenchUniqueName"] = this.componentParameters["TestBenchUniqueName"];

                            cyPhy2Modelica.InvokeEx(project, tbref.Referred, selectedobjs, param);

                            lib_package_folder = cyPhy2Modelica.ComponentParameter["lib_package_folder"] as List<string>;
                            lib_package_folder =
                                lib_package_folder == null ? new List<string>() : lib_package_folder;

                            lib_package_name = cyPhy2Modelica.ComponentParameter["lib_package_name"] as List<string>;
                            lib_package_name =
                                lib_package_name == null ? new List<string>() : lib_package_name;

                            this.componentParameters["labels"] = cyPhy2Modelica.ComponentParameter["labels"] + "&&PCC" + META.VersionInfo.PCC;
                        }
                    }
                });
                MgaGateway.PerformInTransaction(delegate
                {
                    Main(project, currentobj, selectedobjs, Convert(param));
                });
            }
            catch (Exception ex)
            {
                Trace.TraceError(ex.ToString());
            }
            finally
            {
                if (MgaGateway.territory != null)
                {
                    MgaGateway.territory.Destroy();
                }
                MgaGateway = null;
                project = null;
                currentobj = null;
                selectedobjs = null;
                GMEConsole = null;
                GC.Collect();
                GC.WaitForPendingFinalizers();
                System.IO.Directory.SetCurrentDirectory(currentWorkDir);
            }

            META.Logger.RemoveFileListener(this.ComponentName);
        }

        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            object value;
            if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            if (componentParameters == null)
            {
                componentParameters = new SortedDictionary<string, object>();
            }

            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();

        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion


    }
}