/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics.Contracts;
using System.IO;

using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using System.Text.RegularExpressions;
using System.Diagnostics;


namespace CyPhyPET
{
    public class PET
    {
        private CyPhy.ParametricExploration Pet { get; set; }
        public enum TypeOfPET { PCC, Optimizer, ParameterStudy };
        private string OutputDirectory { get; set; }
        public string RelativeSubDirectory { get; set; }
        public bool HasModelicaTestBench { get; set; }
        public List<string> LibPackageFolder { get; set; }
        public List<string> LibPackageName { get; set; }
        public ManifestInfo Manifest { get; set; }

        public static GME.CSharp.GMEConsole GMEConsole { get; set; }

        public string RunCommand { get; set; }

        public PET(
            CyPhy.ParametricExploration pet,
            string outputDirectory,
            List<string> lib_package_folder,
            List<string> lib_package_name)
        {
            Contract.Requires(pet != null);

            Pet = pet;
            OutputDirectory = outputDirectory;
            LibPackageFolder = lib_package_folder;
            LibPackageName = lib_package_name;

            Uri uri1 = new Uri(OutputDirectory + "\\");
            Uri uri2 = new Uri(Directory.GetCurrentDirectory() + "\\");
            RelativeSubDirectory = uri2.MakeRelativeUri(uri1).ToString();
            Manifest = new ManifestInfo();
            Manifest.Author = pet.Impl.Project.Author;
            Manifest.Name = pet.Name;
            Manifest.ModelPath = pet.Path;
            Manifest.SourceFile = pet.Impl.Project.ProjectConnStr;
        }

        /// <summary>
        /// Generate python-scripts and code for the PET.
        /// Three main branches depending on the type of PET.
        /// </summary>
        /// <param name="PETType">TypeOfPET enum - {PCC, Optimizer, ParameterStudy}</param>
        public void GenerateCode(TypeOfPET PETType)
        {
            // generate command files

            // generate assembly

            // generate drivers/optimizers

            if (PETType == TypeOfPET.Optimizer)
            {
                GenerateOptimizerScripts();
            }

            
            if (PETType == TypeOfPET.ParameterStudy)
            {
                GenerateParameterStudyScripts();
            }


            if (PETType == TypeOfPET.PCC)
            {
                GeneratePCCScripts();
            }

            // build workflow
            List<CyPhy.TestBenchRef> tbref = Pet.Children.TestBenchRefCollection.ToList();

            List<CyPhy.TestBenchRef> workflow = new List<CyPhy.TestBenchRef>();
            foreach (var item in Pet.Children.VariableSweepCollection)
            {
                var tb = tbref.FirstOrDefault(x => x.ID == item.GenericDstEndRef.ID);
                if (workflow.FirstOrDefault(x => x.ID == tb.ID) == null)
                {
                    workflow.Add(tb);
                    InsertDependency(workflow, tb);
                }
            }

            var wokflowNames = workflow.Select(x => x.Name).ToList();


            List<CyPhy.TestBenchRef> workflow2 = new List<CyPhy.TestBenchRef>();
            foreach (var item in Pet.Children.ObjectiveMappingCollection)
            {
                var tb = tbref.FirstOrDefault(x => x.ID == item.GenericSrcEndRef.ID);
                if (workflow2.FirstOrDefault(x => x.ID == tb.ID) == null)
                {
                    workflow2.Add(tb);
                    InsertDependency2(workflow2, tb);
                }
            }

            var wokflowNames2 = workflow2.Select(x => x.Name).ToList();


            // generate components
            foreach (var item in Pet.Children.TestBenchRefCollection)
            {
                var testbench = item.Referred.TestBench;

                if (testbench.Attributes.Type == CyPhyClasses.TestBench.AttributesClass.Type_enum.Simple_Calculation)
                {
                    SimpleCalculation(testbench);
                }
                else if (testbench.Attributes.Type == CyPhyClasses.TestBench.AttributesClass.Type_enum.Excel_Spreadsheet)
                {
                    Excel(testbench);
                }
                else if (testbench.Attributes.Type == CyPhyClasses.TestBench.AttributesClass.Type_enum.Dynamics_Simulation)
                {
                    var task = testbench.Children.
                        WorkflowRefCollection.FirstOrDefault().Referred.
                        Workflow.Children.TaskCollection.FirstOrDefault();

                    if (task.Attributes.COMName == "MGA.Interpreter.CyPhyDynamics")
                    {
                        MatLab(testbench);
                    }
                    else if (task.Attributes.COMName == "MGA.Interpreter.CyPhy2Modelica")
                    {
                        Modelica(testbench);
                    }
                    else
                    {
                        GMEConsole.Error.WriteLine("{0} is not supported.", task.Attributes.COMName);
                    }
                }

            }
        }

        private void GeneratePCCScripts()
        {
            var PCCDriver = Pet.Children.PCCDriverCollection.FirstOrDefault();
            string PCCDriverName = ReplaceSpecialChars(PCCDriver.Name);
            string JSONconfigs = PCCDriverName + "configs.json";

            // Generate PCC Driver

            var testbench = Pet.Children.TestBenchRefCollection.FirstOrDefault().Referred.TestBench;

            string modelicaFileName = testbench.Attributes.SpreadsheetLoc;
            string modelicaModelName = (testbench.Impl as GME.MGA.MgaFCO).RegistryValue["ModelName"];
            modelicaModelName = string.IsNullOrEmpty(modelicaModelName) ?
                Path.GetFileNameWithoutExtension(modelicaFileName) :
                modelicaModelName;


            System.Uri uri1 = new Uri(OutputDirectory + "/", UriKind.RelativeOrAbsolute);
            System.Uri uri2 = new Uri(testbench.Attributes.SpreadsheetLoc, UriKind.RelativeOrAbsolute);

            if (uri2.IsAbsoluteUri)
            {
                Uri relativeUri = uri1.MakeRelativeUri(uri2);
                modelicaFileName = relativeUri.ToString();
            }
            else
            {
                uri2 = new Uri(
                    Path.Combine(Directory.GetCurrentDirectory(), testbench.Attributes.SpreadsheetLoc),
                    UriKind.RelativeOrAbsolute);
                Uri relativeUri = uri1.MakeRelativeUri(uri2);
                modelicaFileName = relativeUri.ToString();
            }

            var upMethod = Pet.Children.PCCDriverCollection.FirstOrDefault().Attributes.PCC_UP_Methods;

            Dictionary<CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum, int> upMethodMap =
                new Dictionary<CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum, int>()
                {
                    //{CyPhyClasses.PCCDriver.AttributesClass.PCCMethods_enum.Kriging_method, 0},  //  
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum.Monte_Carlo_Simulation__UP_MCS_, 1},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum.Taylor_Series_Approximation__UP_TS_, 2},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum.Most_Probable_Point_Method__UP_MPP_, 3},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum.Full_Factorial_Numerical_Integration__UP_FFNI_, 4},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum.Univariate_Dimension_Reduction_Method__UP_UDR_, 5},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_UP_Methods_enum.Polynomial_Chaos_Expansion__UP_PCE_, 6}
                };

            var upMethodNum = upMethodMap[upMethod];

            var saMethod = Pet.Children.PCCDriverCollection.FirstOrDefault().Attributes.PCC_SA_Methods;

            if (saMethod == CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum.Morris_methhod__SA_MORRIS_)
            {
                GMEConsole.Error.WriteLine("Morris Method is not yet supported.");
            }

            Dictionary<CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum, int> saMethodMap =
                new Dictionary<CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum, int>()
                {
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum.None, 0},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum.Sobol_method__SA_SOBOL_, 7},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum.Morris_methhod__SA_MORRIS_, 8},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum.FAST_method__SA_FAST_, 9},
                    {CyPhyClasses.PCCDriver.AttributesClass.PCC_SA_Methods_enum.EFAST_method__SA_EFAST_, 10},
                };

            var saMethodNum = saMethodMap[saMethod];


            Templates.PCCopenMDAO pccOMDAO = new Templates.PCCopenMDAO()
            {
                pet = Pet,
                //PCC_CorePath = CyPhyPETInterpreter.META_PATH_PCC_CORE,
                OutputDirectory = OutputDirectory,
                PathToModelicaModel = @"..\\",
                InputJson = JSONconfigs,
                ModelicaFileName = modelicaFileName,
                ModelName = modelicaModelName,
                ModelicaTestBench = HasModelicaTestBench,
            };

            using (StreamWriter writer =
                new StreamWriter(Path.Combine(OutputDirectory, PCCDriverName + ".py")))
            {
                writer.WriteLine(pccOMDAO.TransformText());
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, PCCDriverName + ".py"));

            Templates.runDriver runDriver = new Templates.runDriver()
            {
                DriverName = PCCDriverName
            };

            using (StreamWriter writer =
                new StreamWriter(Path.Combine(OutputDirectory, "run_PCC.py")))
            {
                writer.WriteLine(runDriver.TransformText());
            }

            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, "run_PCC.py"));
            RunCommand = "run_PCC.py";

            using (StreamWriter writer = new StreamWriter(Path.Combine(RelativeSubDirectory, "version_check.py")))
            {
                writer.WriteLine(CyPhyPET.Properties.Resources.version_check);
            }

            using (StreamWriter writer = new StreamWriter(Path.Combine(RelativeSubDirectory, "driver_runner.py")))
            {
                writer.WriteLine(CyPhyPET.Properties.Resources.PCC_runner);
            }

            // Generate input config file for OSU code

            //==========================================================================================
            PCC.Configurations configs = new PCC.Configurations();
            configs.Configuration = new PCC.Configuration();


            configs.Configuration.Parts = new List<PCC.Part>();
            PCC.Part part = new PCC.Part();
            part.ModelConfigFileName = testbench.Name + "_model_config.json";
            part.ToolConfigFileName = testbench.Name + "_tool_config.json";
            configs.Configuration.Parts.Add(part);


            configs.Configuration.Name = PCCDriverName;
            configs.Configuration.ID = PCCDriver.ID;
            configs.Configuration.PCCInputArguments = new PCC.PCCInputArguments();
            var pccInputArgs = configs.Configuration.PCCInputArguments;
            pccInputArgs.InputIDs = new List<string>();
            pccInputArgs.OutputIDs = new List<string>();
            pccInputArgs.StochasticInputs = new PCC.StochasticInputs();
            pccInputArgs.StochasticInputs.InputDistributions = new List<PCC.InputDistribution>();
            pccInputArgs.PCCMetrics = new List<PCC.PCCMetric>();

            foreach (var item in PCCDriver.Children.PCCParameterCollection)
            {
                pccInputArgs.InputIDs.Add(item.ID);
                PCC.InputDistribution inputDist = new PCC.InputDistribution();

                if (item is CyPhy.PCCParameterNormal)
                {
                    inputDist.Distribution = "NORM";
                    inputDist.Param1 = (item as CyPhy.PCCParameterNormal).Attributes.Mean;
                    inputDist.Param2 = (item as CyPhy.PCCParameterNormal).Attributes.StandardDeviation;
                }
                else if (item is CyPhy.PCCParameterUniform)
                {
                    inputDist.Distribution = "UNIF";
                    inputDist.Param1 = (item as CyPhy.PCCParameterUniform).Attributes.LowLimit;
                    inputDist.Param2 = (item as CyPhy.PCCParameterUniform).Attributes.HighLimit;
                }
                else if (item is CyPhy.PCCParameterLNormal)
                {
                    inputDist.Distribution = "LNORM";
                    inputDist.Param1 = (item as CyPhy.PCCParameterLNormal).Attributes.Shape;
                    inputDist.Param2 = (item as CyPhy.PCCParameterLNormal).Attributes.LogScale;
                }
                else if (item is CyPhy.PCCParameterBeta)
                {
                    inputDist.Distribution = "BETA";
                    inputDist.Param1 = (item as CyPhy.PCCParameterBeta).Attributes.Shape1;
                    inputDist.Param2 = (item as CyPhy.PCCParameterBeta).Attributes.Shape2;
                    inputDist.Param3 = (item as CyPhy.PCCParameterBeta).Attributes.LowLimit;
                    inputDist.Param4 = (item as CyPhy.PCCParameterBeta).Attributes.HighLimit;
                }

                foreach (var driveParameter in item.DstConnections.DriveParameterCollection)
                {
                    inputDist.TestBenchParameterNames.Add(driveParameter.DstEnd.Name);
                }

                //inputDist.ID = inputDist.ID == null ? item.ID : inputDist.ID;
                inputDist.ID = item.ID;
                inputDist.Name = item.Name;
                pccInputArgs.StochasticInputs.InputDistributions.Add(inputDist);
            }

            foreach (var item in PCCDriver.Children.PCCOutputCollection)
            {
                pccInputArgs.OutputIDs.Add(item.ID);

                if (item.Attributes.TargetPCCValue == 0)
                {
                    GMEConsole.Warning.WriteLine(
                        "{0}: TargetPCCValue is set to zero. Is this correct?",
                        item.ToHyperLink());
                }

                var pccMetric = new PCC.PCCMetric();
                pccMetric.ID = item.ID;
                pccMetric.Name = item.Name;
                pccMetric.PCC_Spec = item.Attributes.TargetPCCValue;  // Could this ever present a problem?

                foreach (var outputMapping in item.SrcConnections.PCCOutputMappingCollection)
                {
                    pccMetric.TestBenchMetricName = outputMapping.SrcEnd.Name;
                }


                var metricLimits = new PCC.Limits();

                double dMin = double.NegativeInfinity;
                double dMax = double.PositiveInfinity;

                if (double.TryParse(item.Attributes.MinValue, out dMin) == false)
                {
                    // using min infinity
                    GMEConsole.Warning.WriteLine(
                        "{0}: MinValue must be 'Infinity,' '-Infinity,' or a double.",
                        item.ToHyperLink());
                }

                if (double.TryParse(item.Attributes.MaxValue, out dMax) == false)
                {
                    // using max infinity
                    GMEConsole.Warning.WriteLine(
                        "{0}: MaxValue must be 'Infinity,' '-Infinity,' or a double.",
                        item.ToHyperLink());
                }

                metricLimits.Min = dMin;
                metricLimits.Max = dMax;
                metricLimits.op = null; // "min/max/avg/none";       //
                metricLimits.minrange = null; // "value or n/a";     // TODO: add these 3 attributes to the PCCOutput component in CyPhy
                metricLimits.maxrange = null; //"value or n/a";     //
                pccMetric.Limits = metricLimits;

                pccInputArgs.PCCMetrics.Add(pccMetric);
            }

            pccInputArgs.Methods = new List<int>();
            if (upMethodNum > 0)
            {
                pccInputArgs.Methods.Add(upMethodNum);
            }

            if (saMethodNum > 0)
            {
                pccInputArgs.Methods.Add(saMethodNum);
            }


            PCC.RootObject rootConfig = new PCC.RootObject()
            {
                Configurations = configs
            };

            string s = Newtonsoft.Json.JsonConvert.SerializeObject(rootConfig, Newtonsoft.Json.Formatting.Indented);
            using (StreamWriter writer1 = new StreamWriter(Path.Combine(OutputDirectory, JSONconfigs)))
            {
                writer1.WriteLine(s);
            }
        }

        private void GenerateParameterStudyScripts()
        {
            var parameterStudy = Pet.Children.ParameterStudyCollection.FirstOrDefault();
            string parameterStudyName = ReplaceSpecialChars(parameterStudy.Name);

            Dictionary<CyPhyClasses.ParameterStudy.AttributesClass.DOEType_enum, string> doe_map =
                new Dictionary<CyPhyClasses.ParameterStudy.AttributesClass.DOEType_enum, string>()
                        {
                        {CyPhyClasses.ParameterStudy.AttributesClass.DOEType_enum.Central_Composite, "CentralComposite"},
                        {CyPhyClasses.ParameterStudy.AttributesClass.DOEType_enum.Full_Factorial, "FullFactorial"},
                        {CyPhyClasses.ParameterStudy.AttributesClass.DOEType_enum.Opt_Latin_Hypercube, "LatinHypercube"},
                        {CyPhyClasses.ParameterStudy.AttributesClass.DOEType_enum.Uniform, "Uniform"}
                    };

            Dictionary<CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum, string> surrogate_map =
                new Dictionary<CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum, string>()
                        {
                        {CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum.Kriging_Surrogate, "KrigingSurrogate"},
                        {CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum.Logistic_Regression, "LogisticRegression"},
                        {CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum.Neural_Net, "NeuralNet"},
                        {CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum.None, ""},
                        {CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum.Response_Surface, "ResponseSurface"}
                    };

            if (parameterStudy.Attributes.SurrogateType == CyPhyClasses.ParameterStudy.AttributesClass.SurrogateType_enum.None)
            {
                Templates.DOE Doe = new Templates.DOE();
                Doe.DOEType = doe_map[parameterStudy.Attributes.DOEType];
                Doe.pet = Pet;
                Doe.OutputDirectory = OutputDirectory;
                string value = Doe.TransformText();
                using (StreamWriter writer =
                    new StreamWriter(Path.Combine(OutputDirectory, Pet.Name + "_called.py")))
                {
                    writer.WriteLine(value);
                }
                Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, Pet.Name + "_called.py"));
            }
            else
            {
                // SurrogateAssembly.tt
                Templates.SurrogateAssembly SurrogateAsm = new Templates.SurrogateAssembly();
                SurrogateAsm.ParameterStudyName = parameterStudyName;
                SurrogateAsm.pet = Pet;
                SurrogateAsm.OutputDirectory = OutputDirectory;
                string value = SurrogateAsm.TransformText();
                using (StreamWriter writer =
                    new StreamWriter(Path.Combine(OutputDirectory, parameterStudyName + ".py")))
                {
                    writer.WriteLine(value);
                }
                Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, parameterStudyName + ".py"));

                //Surrogate.tt
                Templates.Surrogate Surrogate = new Templates.Surrogate();
                Surrogate.DOEType = doe_map[parameterStudy.Attributes.DOEType];
                Surrogate.SurrogateType = surrogate_map[parameterStudy.Attributes.SurrogateType];
                Surrogate.pet = Pet;
                Surrogate.OutputDirectory = OutputDirectory;
                Surrogate.ParameterStudyName = parameterStudyName;

                using (StreamWriter writer =
                    new StreamWriter(Path.Combine(OutputDirectory, Pet.Name + "_called.py")))
                {
                    writer.WriteLine(Surrogate.TransformText());
                }
                Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, Pet.Name + "_called.py"));

                //SurrogateValidation.tt
                Templates.SurrogateValidation SurrogateValidation = new Templates.SurrogateValidation();
                SurrogateValidation.ParameterStudyName = parameterStudyName;
                SurrogateValidation.pet = Pet;
                SurrogateValidation.OutputDirectory = OutputDirectory;
                string value3 = SurrogateValidation.TransformText();
                using (StreamWriter writer =
                    new StreamWriter(Path.Combine(OutputDirectory, Pet.Name + "_validation.py")))
                {
                    writer.WriteLine(value3);
                }
                Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, Pet.Name + "_validation.py"));

                //runDriver.tt
                Templates.runDriver runValidation = new Templates.runDriver()
                {
                    DriverName = Pet.Name + "_validation",
                };
                using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, "run_parameter_study_validation.py")))
                {
                    writer.WriteLine(runValidation.TransformText());
                }
                Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, "run_parameter_study_validation.py"));

            }

            /// Resource Files (non-templates)
            //version_check.py
            using (StreamWriter writer = new StreamWriter(Path.Combine(RelativeSubDirectory, "version_check.py")))
            {
                writer.WriteLine(CyPhyPET.Properties.Resources.version_check);
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, "version_check.py"));


            //driver_runner.py
            using (StreamWriter writer = new StreamWriter(Path.Combine(RelativeSubDirectory, "driver_runner.py")))
            {
                writer.WriteLine(CyPhyPET.Properties.Resources.PCC_runner);
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, "driver_runner.py"));
            
            
            /// Main entry point for executing PET
            //runDriver.tt
            Templates.runDriver runDOE = new Templates.runDriver()
            {
                DriverName = Pet.Name + "_called"
            };

            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, "run_parameter_study.py")))
            {
                writer.WriteLine(runDOE.TransformText());
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, "run_parameter_study.py"));
            RunCommand = "run_parameter_study.py";
            Manifest.EntryPoint = Path.Combine(RelativeSubDirectory, "run_parameter_study.py");
        }

        private void GenerateOptimizerScripts()
        {
            var optimizer = Pet.Children.OptimizerCollection.FirstOrDefault();

            Dictionary<CyPhyClasses.Optimizer.AttributesClass.OptimizationFunction_enum, string> optimizer_map =
                new Dictionary<CyPhyClasses.Optimizer.AttributesClass.OptimizationFunction_enum, string>()
                    {
                        {CyPhyClasses.Optimizer.AttributesClass.OptimizationFunction_enum.COBYLA, "COBYLAdriver"},
                        {CyPhyClasses.Optimizer.AttributesClass.OptimizationFunction_enum.CONMIN, "CONMINdriver"},
                        {CyPhyClasses.Optimizer.AttributesClass.OptimizationFunction_enum.NEWSUMT, "NEWSUMTdriver"}
                    };

            Templates.Optimizer Optimizer = new Templates.Optimizer();
            Optimizer.OptimizerType = optimizer_map[optimizer.Attributes.OptimizationFunction];
            Optimizer.pet = Pet;
            Optimizer.OutputDirectory = OutputDirectory;
            string value = Optimizer.TransformText();
            using (StreamWriter writer =
                new StreamWriter(Path.Combine(OutputDirectory, Pet.Name + ".py")))
            {
                writer.WriteLine(value);
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, Pet.Name + ".py"));
            Templates.runDriver runDOE = new Templates.runDriver()
            {
                DriverName = Pet.Name
            };

            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, "run_" + Pet.Name + ".py")))
            {
                writer.WriteLine(runDOE.TransformText());
                //writer.WriteLine("echo off");
                //writer.WriteLine("set PATH=%PATH%;%OPENMODELICAHOME%\\MinGW\\bin");
                //writer.WriteLine("\"C:\\OpenMDAO\\openmdao-0.2.5\\Scripts\\python.exe\" {0}", Pet.Name + ".py");
            }
            using (StreamWriter writer = new StreamWriter(Path.Combine(RelativeSubDirectory, "version_check.py")))
            {
                writer.WriteLine(CyPhyPET.Properties.Resources.version_check);
            }

            using (StreamWriter writer = new StreamWriter(Path.Combine(RelativeSubDirectory, Pet.Name + "_runner.py")))
            {
                writer.WriteLine(CyPhyPET.Properties.Resources.PCC_runner);
            }

            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, "run_" + Pet.Name + ".py"));
            Manifest.EntryPoint = Path.Combine(RelativeSubDirectory, "run_" + Pet.Name + ".py");
            //Manifest.EntryPoint = Path.Combine(RelativeSubDirectory, Pet.Name + ".py");
        }

        private void InsertDependency(List<CyPhy.TestBenchRef> workflow, CyPhy.TestBenchRef tb)
        {
            var outConns = Pet.Children.ResultFlowCollection.Where(x => x.GenericSrcEndRef.ID == tb.ID);
            bool cyclic = false;

            foreach (var item in outConns)
            {
                var nexttb = Pet.Children.TestBenchRefCollection.FirstOrDefault(x => item.GenericDstEndRef.ID == x.ID);

                if (Pet.Children.ResultFlowCollection.Count() != workflow.Count)
                {
                    var tbinlist = workflow.FirstOrDefault(x => x.ID == tb.ID);
                    int i = workflow.IndexOf(tbinlist);
                    i++;
                    if (workflow.FirstOrDefault(x => x.ID == nexttb.ID) == null)
                    {
                        workflow.Insert(i, nexttb);
                        InsertDependency(workflow, nexttb);
                    }
                    else
                    {
                        // break loop!
                        //throw new Exception("cyclic dependency was detected among testbenchrefs");
                        cyclic = true;
                        break;
                    }
                }
            }
        }


        private void InsertDependency2(List<CyPhy.TestBenchRef> workflow, CyPhy.TestBenchRef tb)
        {
            var outConns = Pet.Children.ResultFlowCollection.Where(x => x.GenericDstEndRef.ID == tb.ID);
            bool cyclic = false;

            foreach (var item in outConns)
            {
                var nexttb = Pet.Children.TestBenchRefCollection.FirstOrDefault(x => item.GenericSrcEndRef.ID == x.ID);

                if (Pet.Children.TestBenchRefCollection.Count() != workflow.Count)
                {
                    var tbinlist = workflow.FirstOrDefault(x => x.ID == tb.ID);
                    int i = workflow.IndexOf(tbinlist);
                    i++;
                    if (workflow.FirstOrDefault(x => x.ID == nexttb.ID) == null)
                    {
                        workflow.Insert(i, nexttb);
                        InsertDependency2(workflow, nexttb);
                    }
                    else
                    {
                        // break loop!
                        //throw new Exception("cyclic dependency was detected among testbenchrefs");
                        cyclic = true;
                        break;
                    }
                }
            }
        }


        private void Modelica(CyPhy.TestBench testbench)
        {
            string toolName = "OpenModelica";

            // TODO: we need a better implementation here based on the requirements.
            if (testbench.Children.WorkflowRefCollection.Count() == 1)
            {
                var workflowRef = testbench.Children.WorkflowRefCollection.FirstOrDefault();
                var wf = workflowRef.Referred.Workflow;
                if (wf != null)
                {
                    var task = wf.Children.TaskCollection.FirstOrDefault();
                    if (task != null)
                    {
                        if (task.Attributes.COMName == "MGA.Interpreter.CyPhy2Modelica")
                        {
                            try
                            {
                                var parameters = Newtonsoft.Json.Linq.JObject.Parse(task.Attributes.Parameters);
                                string stt = parameters["simulationTargetTool"].ToString();
                                toolName = stt == null ? toolName : stt;
                            }
                            catch
                            {
                                // ignore
                            }
                        }
                    }
                }
            }

            Templates.Modelica modelica = new Templates.Modelica(){
                testBench = testbench,
                OutputDirectory = OutputDirectory,
                toolName = toolName
            };
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, testbench.Name + ".py")))
            {
                writer.WriteLine(modelica.TransformText());
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, testbench.Name + ".py"));
            Manifest.ResourceFiles.AddRange(modelica.GetResourceFileNames());

            List<string> libPaths = new List<string>();
            for (int i = 0; i < LibPackageFolder.Count; i++)
            {
                libPaths.Add(Path.Combine(LibPackageFolder.ElementAt(i), LibPackageName.ElementAt(i)));
            }
            Templates.zip zippy = new Templates.zip()
            {
                FullPackagePaths = libPaths,
                OutputFileName = "source_data.zip",
                TestBenchName = testbench.Name
            };
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, "zip.py")))
            {
                writer.WriteLine(zippy.TransformText());
            }

            var modelFileNameConfig = new Dictionary<string, string>()
            {
                {"model_file_name", testbench.Name + "." + testbench.Name}
            };

            // Do not change the output file name. It is being used elsewhere in this code.
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, testbench.Name + "_model_config.json")))
            {
                writer.WriteLine(Newtonsoft.Json.JsonConvert.SerializeObject(modelFileNameConfig, Newtonsoft.Json.Formatting.Indented));
            }
        }

        private void SimpleCalculation(CyPhy.TestBench testbench)
        {
            Templates.SimpleCalculation m = new Templates.SimpleCalculation();
            m.testBench = testbench;
            m.OutputDirectory = OutputDirectory;
            string value = m.TransformText();
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, testbench.Name + ".py")))
            {
                writer.WriteLine(value);
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, testbench.Name + ".py"));
        }

        private void Excel(CyPhy.TestBench testbench)
        {
            Templates.Excel m = new Templates.Excel();
            m.testBench = testbench;
            m.OutputDirectory = OutputDirectory;
            string value = m.TransformText();
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, testbench.Name + ".py")))
            {
                writer.WriteLine(value);
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, testbench.Name + ".py"));
            Manifest.ResourceFiles.AddRange(m.GetResourceFileNames());
        }

        private void MatLab(CyPhy.TestBench testbench)
        {
            Templates.MatLab m = new Templates.MatLab();
            m.testBench = testbench;
            m.OutputDirectory = OutputDirectory;
            string value = m.TransformText();
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, testbench.Name + ".py")))
            {
                writer.WriteLine(value);
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, testbench.Name + ".py"));
            Manifest.ResourceFiles.AddRange(m.GetResourceFileNames());
        }

        private void JModelica(CyPhy.TestBench testbench)
        {
            Templates.JModelica m = new Templates.JModelica();
            m.testBench = testbench;
            m.OutputDirectory = OutputDirectory;
            string value = m.TransformText();
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, testbench.Name + ".py")))
            {
                writer.WriteLine(value);
            }
            Manifest.ResourceFiles.Add(Path.Combine(RelativeSubDirectory, testbench.Name + ".py"));
            Manifest.ResourceFiles.AddRange(m.GetResourceFileNames());
        }

        public static string ReplaceSpecialChars(string inputString)
        {
            StringBuilder sb = new StringBuilder();
            bool didChange = false;
            foreach (char c in inputString)
            {
                if ((c >= '0' && c <= '9') || (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || c=='_')
                {
                    sb.Append(c);
                }
                else
                {
                    sb.Append('_');
                    didChange = true;
                }
            }

            if (sb[0] >= '0' && sb[0] <= '9')
            {
                sb[0] = '_';
                didChange = true;
            }
            string returnString = sb.ToString();

            if (didChange)
            {
                Trace.TraceInformation("Name {0} contains illegal characters. In generated code the name will be {1}.", inputString, returnString);
            }

            return returnString;
        }

    }

    #region Extension Methods
    public static class GmeConsoleHelper
    {
        public static string ToHyperLink<T>(this T subject)
            where T : ISIS.GME.Common.Interfaces.Base
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("<a href=\"mga:{0}\">{1}</a>", subject.ID, subject.Name);
            return sb.ToString();
        }

        public static string ToMgaHyperLink<T>(this T subject)
            where T : GME.MGA.MgaFCO
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("<a href=\"mga:{0}\">{1}</a>", subject.ID, subject.Name);
            return sb.ToString();
        }
    }
    #endregion
}
