import os
import sys
import json
import time
import datetime
import multiprocessing
import platform
import subprocess

LINUX_VIR_ENV = '~/openmdao-0.2.7/'

def _query_registry():
    """
    Queries the Windows registry for META_PATH in order to get the location of
    python virtual environment containing all necessary packages and modules. 
    """
    import _winreg as wr
    try:
        # try to obtain META_PATH from the registry
        key = wr.OpenKey(wr.HKEY_LOCAL_MACHINE, r'software\meta', 0, wr.KEY_READ)
        meta_path = wr.QueryValueEx(key, 'META_PATH')[0]
        py_path = os.path.join(meta_path, r'bin\Python27\Scripts\python')
    except WindowsError:
        sys.stderr.write('Could not find META_PATH in registry, attempting to use default python.')
        py_path = 'python'
    
    return py_path


def _write_out_stat(execution_time, iso_time):
    """
    Writes out statistics to stat.json
    """
    dyn_stat = {"total_time": execution_time, "start_time": iso_time}
    try:
        with open('stat.json', 'r') as f_in:
            stat = json.load(f_in)
        if stat['ToolSpecifics'] is None:
            stat['ToolSpecifics'] = {'dynamics': dyn_stat}
        else:
            stat['ToolSpecifics']['dynamics'] = dyn_stat
    except (IOError, KeyError):
        stat = {'dynamics': dyn_stat}

    try:
        with open('tool_stat.json', 'r') as f_in:
            tool_stat = json.load(f_in)
        os.remove('tool_stat.json')
    except IOError:
        tool_stat = {}
        
    dyn_stat['tool_stat'] = tool_stat
    
    dyn_stat['is_pcc'] = True
    dyn_stat['cpu_count'] = multiprocessing.cpu_count()
    dyn_stat['system'] = platform.system()
    dyn_stat['machine'] = platform.machine()
    dyn_stat['processor'] = platform.processor()
    
    if os.name == 'nt':
        dyn_stat['computer_name'] = os.environ['COMPUTERNAME']
        dyn_stat['win32_ver'] = platform.win32_ver()
    else:
        dyn_stat['COMPUTER_NAME'] = "usr"
        dyn_stat['win32_ver'] = []

    with open('stat.json', 'wb') as f_out:
        json.dump(stat, f_out, indent=4)


def _copy_log_files(dir_name='log'):
    """
    Gathers all files that ends with either .log or .txt, prepends dyn_ to the file-name
    and copies them into a log-dir

    """
    import glob
    import shutil
    if not os.path.isdir(dir_name):
        os.mkdir(dir_name)
    try:
        with open('summary.testresults.json', 'r') as f_in:
            dynamic_dir = json.load(f_in)['DesignName']
        cg_log_files = glob.glob('{0}{1}*cg.log'.format(dynamic_dir, os.path.sep))
        for f_src in cg_log_files:
            f_dst = os.path.join(dir_name, "dynamics_{0}".format(os.path.basename(f_src)))
            shutil.copy2(f_src, f_dst)
    except IOError and KeyError:
        print 'When copying log-files : Could not find summary.testresults.json or DesignName-item.'


def run_driver(driver_name):
    t_1 = time.time()
    iso_time = datetime.datetime.now().isoformat()
    if os.name == 'nt':                 
        py_path = _query_registry()
        command = '"{0}" {1}'.format(py_path, driver_name)
    elif os.name == 'posix':
        py_exc = os.path.join(LINUX_VIR_ENV, '/bin/python')
        command = '{0} {1}'.format(py_exc, driver_name)
    else:
        print 'ERROR: Only Windows or Linux supported.'
        _copy_log_files()
        sys.exit(5)

    # call and run python script
    print 'Calling "{0}" as a subprocess.'.format(command)
    try:
        popen = subprocess.Popen(command, shell=True)
        (stdoutdata, _) = popen.communicate()
        #print stdoutdata
        #popen.wait()
        if popen.returncode != 0:
            raise subprocess.CalledProcessError(popen.returncode, command)
            
        execution_time = time.time() - t_1
        _write_out_stat(execution_time, iso_time)
        _copy_log_files()
    except subprocess.CalledProcessError, err:
        print 'Outprint : {0}\n\n{1}'.format(err, err.output)
        print 'Failed calling {0}'.format(command)
        _copy_log_files()
        sys.exit(5)