/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/

namespace CyPhyPrepareIFab
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Runtime.InteropServices;
    using System.Text;
    using System.Linq;
    using System.Diagnostics;

    using GME.CSharp;
    using GME;
    using GME.MGA;
    using GME.MGA.Core;
    using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
    using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
    using Newtonsoft.Json;

    /// <summary>
    /// This class implements the necessary COM interfaces for a GME interpreter component.
    /// </summary>
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public partial class CyPhyPrepareIFabInterpreter : IMgaComponentEx, IGMEVersionInfo
    {
        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        public enum ComponentStartMode
        {
            GME_MAIN_START = 0, 		// Not used by GME
            GME_BROWSER_START = 1,      // Right click in the GME Tree Browser window
            GME_CONTEXT_START = 2,		// Using the context menu by right clicking a model element in the GME modeling window
            GME_EMBEDDED_START = 3,		// Not used by GME
            GME_MENU_START = 16,		// Clicking on the toolbar icon, or using the main menu
            GME_BGCONTEXT_START = 18,	// Using the context menu by right clicking the background of the GME modeling window
            GME_ICON_START = 32,		// Not used by GME
            GME_SILENT_MODE = 128 		// Not used by GME, available to testers not using GME
        }

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a tansaction.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        public void Initialize(MgaProject project)
        {
            // TODO: Add your initialization code here... 
            this.componentParameters = new SortedDictionary<string, object>();
            this.componentParameters.Add("console_messages", "on");
            this.componentParameters.Add("output_dir", Path.Combine(Environment.CurrentDirectory, "results"));
            this.componentParameters.Add("automation", "false");
            this.componentParameters.Add("do_config", "false");
            this.componentParameters.Add("runCommand", "PrepareIFab.bat");
            this.componentParameters.Add("build_query", "componentPackageRequestList");
            this.componentParameters.Add("labels", "Creo&&CADCreoParametricCreateAssembly.exev1.3");
            this.componentParameters.Add("results_zip_py", "");

            this.ManufacturingManifestData = new DesignManufactureManifest();
            this.ComponentManufacturingDataList = new List<ComponentManufacturingData>();
            this.AVMComponentList = new List<AVM.DDP.MetaAvmProject.Component>();
        }

        public string OutputDirectory { get; set; }
        public string AssemblyName { get; set; }
        public string TestBenchName { get; set; }
        public string ArtifactScriptName { get; set; }
        public string RunScriptName { get; set; }
        public string ProjectRootDirectory { get; set; }
        private List<ComponentManufacturingData> ComponentManufacturingDataList { get; set; }
        private DesignManufactureManifest ManufacturingManifestData { get; set; }
        private List<AVM.DDP.MetaAvmProject.Component> AVMComponentList { get; set; }

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is available. A general try-catch block catches all the exceptions
        /// coming from this function, you don't need to add it. For more information, see InvokeEx.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        /// <param name="currentobj">The model open in the active tab in GME. Its value is null if no model is open (no GME modeling windows open). </param>
        /// <param name="selectedobjs">
        /// A collection for the selected model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the context menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the collection is empty (contains zero elements).
        /// </param>
        /// <param name="startMode">Contains information about the GUI event that initiated the invocation.</param>
        [ComVisible(false)]
        public void Main(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, ComponentStartMode startMode)
        {
            // TODO: Add your interpreter code
            //GMEConsole.Out.WriteLine(DateTime.Now.ToString() + " running CyPhyIFab Interpreter");

			// Get RootFolder
			//IMgaFolder rootFolder = project.RootFolder;
			//GMEConsole.Out.WriteLine(rootFolder.Name);       
			
        }

        #region IMgaComponentEx Members

        MgaGateway MgaGateway { get; set; }
        GMEConsole GMEConsole { get; set; }

        public void InvokeEx(MgaProject project, 
                            MgaFCO currentobj, 
                            MgaFCOs selectedobjs, 
                            int param)
        {
            if (!enabled)
            {
                return;
            }

            try
            {                  
                GMEConsole = GMEConsole.CreateFromProject(project);
                MgaGateway = new MgaGateway(project);
                project.CreateTerritoryWithoutSink(out MgaGateway.territory);

                GMEConsole.Out.WriteLine(DateTime.Now.ToString() + " running CyPhyPrepIFab Interpreter");

                this.OutputDirectory = this.componentParameters["output_dir"] as string;
                this.RunScriptName = "PrepareIFab.bat";
                this.ArtifactScriptName = "AppendIFabArtifact.py";

                InitLogger();

                // [1] create avmproj
                string projectName = "";
                MgaGateway.PerformInTransaction(delegate
                {
                    projectName = project.Name;
                },
                transactiontype_enum.TRANSACTION_NON_NESTED);

                string avmProjFileName = Path.Combine(Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length)), "manifest.project.json");
                this.ProjectRootDirectory = Path.GetDirectoryName(avmProjFileName);
                AVM.DDP.MetaAvmProject avmProj = new AVM.DDP.MetaAvmProject();

                if (File.Exists(avmProjFileName))
                {
                    string sjson = "{}";
                    using (StreamReader reader = new StreamReader(avmProjFileName))
                    {
                        sjson = reader.ReadToEnd();
                        avmProj = JsonConvert.DeserializeObject<AVM.DDP.MetaAvmProject>(sjson);
                        this.AVMComponentList = avmProj.Project.Components;
                    }
                }
                // end create avmproj


                // [1] CyPhy2CAD                                 
                // [2] Export DDP, Manufacture XML, Manufacture Manifest
                // [3] Generate AppendArtifact.py - script to append artifacts to summary.testresults.json files
                // [4] Generate DesignModel1BOM.py - script to generate .bom.json from ddp file
                // [5] Generate main run bat file

                MgaGateway.PerformInTransaction(delegate
                {
                    ElaborateModel(project, currentobj, selectedobjs, param);                   // elaborate model                   
                    CallCAD(project, currentobj, selectedobjs, param);                              // CyPhy2CAD                      
                    ManufacturingGeneration(currentobj);                                            // DDP, Manufacture XML, Manufacture Manifest
                },
                transactiontype_enum.TRANSACTION_GENERAL);

                GenerateAppendArtifactScript();                                                     // AppendArtifact.py                                               
                GenerateBOMGenScript();                                                             // DesignModel1BOM.py                                                 
                GenerateRunBatFile();                                                               // main run bat file                                            

                GMEConsole.Out.WriteLine("CyPhyPrepIFab Interpreter Finished!");
            }
            catch (Exception ex)
            {
               GMEConsole.Error.WriteLine("Exception occured: {0}", ex.ToString());
            }
            finally
            {
               if (MgaGateway.territory != null)
                {
                   MgaGateway.territory.Destroy();
                }
                MgaGateway = null;
                project = null;
                currentobj = null;
                selectedobjs = null;
                GMEConsole = null;
                GC.Collect();
                GC.WaitForPendingFinalizers();
            }
            Trace.Flush();
            Trace.Close();
        }

        private void CallCAD(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            int param)
        {
            bool automation = this.componentParameters["automation"] as string == "true";
            Type tCAD = Type.GetTypeFromProgID("MGA.Interpreter.CyPhy2CAD");
            if (tCAD == null)
            {
                GMEConsole.Info.WriteLine("CyPhy2CAD is not installed on your machine.");
                return;
            }
            IMgaComponentEx cadGenerator = Activator.CreateInstance(tCAD) as IMgaComponentEx;

            cadGenerator.ComponentParameter["automation"] = this.componentParameters["automation"];
            cadGenerator.ComponentParameter["prepIFab"] = "true";
            cadGenerator.ComponentParameter["output_dir"] = this.componentParameters["output_dir"];

            if (automation)
            {
                //cadGenerator.ComponentParameter["output_dir"] = this.componentParameters["output_dir"];
                cadGenerator.ComponentParameter["do_config"] = this.componentParameters["do_config"];
                cadGenerator.ComponentParameter["console_messages"] = this.componentParameters["console_messages"];
            }

            this.componentParameters["results_zip_py"] = cadGenerator.ComponentParameter["results_zip_py"];
            cadGenerator.Initialize(currentobj.Project);
            cadGenerator.InvokeEx(currentobj.Project,
                currentobj,
                (MgaFCOs)Activator.CreateInstance(Type.GetTypeFromProgID("Mga.MgaFCOs")),
                param);
        }

        /*
        private void GenerateDDP(MgaFCO currentobj)
        {
            string type = currentobj.MetaBase.Name;

            if (type == "TestBench")
            {
                CyPhy.TestBench tb = CyPhyClasses.TestBench.Cast(currentobj);
                var tlsut = tb.Children.TopLevelSystemUnderTestCollection.FirstOrDefault();
                if (tlsut == null)
                {
                    throw new Exception("There is no top level system under test in the model!");
                }
                if (tlsut.Referred.ComponentAssembly == null)
                {
                    throw new Exception("Top level system under test does not reference a component assembly!");
                }

                var design = CyPhy2DesignInterchange.CyPhy2DesignInterchange.Convert(tlsut.Referred.ComponentAssembly);
                this.TestBenchName = tb.Name;
                this.AssemblyName = design.Name;
                string d = design.SerializeToFile(Path.Combine(this.OutputDirectory, this.TestBenchName + ".metadesign.json"));
            }
            else if (type == "ComponentAssembly")
            {
                CyPhy.ComponentAssembly assembly = CyPhyClasses.ComponentAssembly.Cast(currentobj);
                var design = CyPhy2DesignInterchange.CyPhy2DesignInterchange.Convert(assembly);
                this.AssemblyName = design.Name;
                this.TestBenchName = design.Name;
                string d = design.SerializeToFile(Path.Combine(this.OutputDirectory, this.TestBenchName + ".metadesign.json"));                
            }
            else
                throw new NotImplementedException("");
        }
        */

        private void GenerateRunBatFile()  
        {
            // [1] Invoke .bat to create Metrics.xml and assembly.STEP
            // [2] Invoke BOM_Generator.py
            StreamWriter file = new StreamWriter(Path.Combine(this.OutputDirectory, this.RunScriptName));
            string batfilename = ReplaceSpecialChars(this.TestBenchName);
            file.WriteLine("call " + batfilename + "_Cad.bat");
            file.WriteLine("python DesignModel2BOM.py ");               //file.WriteLine("python DesignModel2BOM.py " + this.AssemblyName + ".metadesign.json " + this.AssemblyName + ".bom.json");
            file.WriteLine("python " + this.ArtifactScriptName);        //file.WriteLine("python " + this.ArtifactScriptName + " " + this.AssemblyName);
            file.Close();
        }

        private string ReplaceSpecialChars(string old)
        {
            char[] chars = new char[] { ',', '.', '=', '(', ')', '!', '@', '#', '$', '%', '&', '~', '`', '+', ' ', '^', '*', '[', ']', '{', '}', '/', '?', ':', ';', '<', '>', '|' };
            string newStr = chars.Aggregate(old, (c1, c2) => c1.Replace(c2, '_'));
            return newStr;
        }

        private void GenerateTestResultsJson(MgaFCO currentobj)
        {
            if ((currentobj.MetaBase.Name == "TestBench") && 
                (this.componentParameters["automation"] as string == "false"))
            {
                /*
                CyPhy.Tes     tBench tb = CyPhyClasses.TestBench.Cast(currentobj);
                AVM.DDP.MetaTBReport report = new AVM.DDP.MetaTBReport();

                report.GenerateSummary(
                    tb.Name,
                    tb,
                    "summary.testresults.json",
                    this.componentParameters["output_dir"] as string);
                */
            }
        }

        private void GenerateAppendArtifactScript()
        {
            Templates.AppendIFabArtifact appendscript = new Templates.AppendIFabArtifact()
            {
                AssemblyName = ReplaceSpecialChars(this.AssemblyName),
                TestBenchName = this.TestBenchName,
                TestBenchName_Cleaned = ReplaceSpecialChars(this.TestBenchName),
            };
            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, this.ArtifactScriptName)))
            {
                writer.WriteLine(appendscript.TransformText());
            }
        }

        private void GenerateBOMGenScript()
        {
            Templates.DesignModel2BOM bomscript = new Templates.DesignModel2BOM()
            {
                DDPFile = this.TestBenchName + ".metadesign.json",
                BOMFile = this.TestBenchName + ".bom.json",
            };
            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "DesignModel2BOM.py")))
            {
                writer.WriteLine(bomscript.TransformText());
            }
        }

        private void ElaborateModel(MgaProject project, 
                                    MgaFCO currentobj, 
                                    MgaFCOs selectedobjs, 
                                    int param)
        {
            // call elaborator and expand the references
            Type t = Type.GetTypeFromProgID("MGA.Interpreter.CyPhyElaborate");
            IMgaComponentEx elaborator = Activator.CreateInstance(t) as IMgaComponentEx;
            elaborator.Initialize(project);
            elaborator.ComponentParameter["automated_expand"] = "true";
            elaborator.ComponentParameter["console_messages"] = "off";

            elaborator.InvokeEx(project, currentobj, selectedobjs, param);
        }


        private void InitLogger()
        {
            string logDir = Path.Combine(this.OutputDirectory, "log");
            if (!Directory.Exists(logDir))
                Directory.CreateDirectory(logDir);

            string loggingFileName = Path.Combine(logDir, "CyPhyPrepareIFab.log");

            var fs = new FileStream(loggingFileName, FileMode.Create);
            TraceListener fileTL = new TextWriterTraceListener(fs)
            {
                Name = "CyPhyPrepareIFab",
            };

            Trace.AutoFlush = true;
            Trace.Listeners.Add(fileTL);
            Trace.TraceInformation("{0} trace file listener was created.", loggingFileName);
        }

        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            object value;
            if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            if (componentParameters == null)
            {
                componentParameters = new SortedDictionary<string, object>();
            }

            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();

        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion


    }
}
