/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "stdafx.h"
#include "UdmApp.h"
#include "UdmConfig.h"
#include "Uml.h"
#include "UdmUtil.h"
#include "UdmConsole.h"

using namespace std;

/*********************************************************************************/
/* Initialization function. The framework calls it before preparing the backend. */
/* Initialize here the settings in the config global object.					 */
/* Return 0 if successful.														 */
/*********************************************************************************/
int CUdmApp::Initialize()
{


	// TODO: Your initialization code comes here...
	return 0;
}



/* 
Remarks to CUdmApp::UdmMain(...):
0.	The p_backend points to an already open backend, and the framework 
	closes it automatically. DO NOT OPEN OR CLOSE IT!
	To commit changes use p_backend->CommitEditSequence().
	To abort changes use p_backend->AbortEditSequence().
	To save changes to a different file use p_backend->SaveAs() or p_backend->CloseAs().

1.	Focus is the currently open model.

2.	The possible values for param (from GME Mga.idl component_startmode_enum):
	GME_MAIN_START			=   0,
	GME_BROWSER_START		=   1,
	GME_CONTEXT_START		=   2,
	GME_EMBEDDED_START		=   3,
	GME_MENU_START			=  16,
	GME_BGCONTEXT_START		=  18,
	GME_ICON_START			=  32,
	METAMODEL_CHECK_SYNTAX	= 101

 3. The framework catches all the exceptions and reports the error in a message box,
	clean up and close the transactions aborting the changes. You can override this 
	behavior by catching udm_exception. Use udm_exception::what() to form an error 
	message.
*/

/***********************************************/
/* Main entry point for Udm-based Interpreter  */
/***********************************************/

#include "C:\Python27\include\Python.h"

typedef PyObject* (__cdecl *Object_Convert_t)(Udm::Object udmObject);

struct PyObject_RAII
{
	PyObject* p;
	PyObject_RAII() : p(NULL) { }
	PyObject_RAII(PyObject* p) : p(p) { }
	operator PyObject*() { return p; }
	~PyObject_RAII() { Py_XDECREF(p); }
};

std::string GetPythonError()
{
	PyObject_RAII type, value, traceback;
	PyErr_Fetch(&type.p, &value.p, &traceback.p);
	PyErr_Clear();
	std::string error;
	if (type)
	{
		PyObject_RAII type_name = PyObject_GetAttrString(type, "__name__");
		if (type_name && PyString_Check(type_name))
		{
			error += PyString_AsString(type_name);
			error += ": ";
		}
	}
	PyObject_RAII message = PyObject_GetAttrString(value, "message");
	if (message && PyString_Check(message))
	{
		PyObject_RAII str_value = PyObject_Str(message);
		error += PyString_AsString(str_value);
	}
	else if (type && value && value.p->ob_type->tp_str)
	{
		PyObject_RAII str = value.p->ob_type->tp_str(value);
		error += PyString_AsString(str);
	}
	else
		error += "Unknown exception";

	error += ": ";
	if (traceback)
	{
		PyObject_RAII main = PyImport_ImportModule("__main__");
		PyObject* main_namespace = PyModule_GetDict(main);
		PyObject_RAII dict = PyDict_Copy(main_namespace);
		PyDict_SetItemString(dict, "tb", traceback);
		PyObject_RAII _none = PyRun_StringFlags(
			"import traceback\n"
			"tb = ''.join(traceback.format_tb(tb))\n", Py_file_input, dict, dict, NULL);
		PyObject* formatted_traceback = PyDict_GetItemString(dict, "tb");
		error += PyString_AsString(formatted_traceback);
	}
	else
		error += "Unknown traceback";
	return error;
}

std::string openfilename(char *filter = "All Files (*.*)\0*.*\0", HWND owner = NULL)
{
	OPENFILENAME ofn;
	char fileName[MAX_PATH] = "";
	ZeroMemory(&ofn, sizeof(ofn));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = owner;
	ofn.lpstrFilter = filter;
	ofn.lpstrFile = fileName;
	ofn.nMaxFile = MAX_PATH;
	ofn.Flags = OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
	ofn.lpstrDefExt = "";
	string fileNameStr;
	if (GetOpenFileNameA(&ofn))
		fileNameStr = fileName;
	return fileNameStr;
}

static PyObject *CyPhyPython_log(PyObject *self, PyObject *args)
{
	PyObject* arg1 = PyTuple_GetItem(args, 0);
	if (PyUnicode_Check(arg1))
	{
		GMEConsole::Console::Out::writeLine(PyUnicode_AsUnicode(arg1));
	}
	else if (PyString_Check(arg1))
	{
		GMEConsole::Console::Out::writeLine(PyString_AsString(arg1));
	}
	else
		return NULL;
	Py_INCREF(Py_None);
	return Py_None;
}


static PyMethodDef CyPhyPython_methods[] = {
    {"log",  CyPhyPython_log, METH_VARARGS, "Log in the GME console"},
    {NULL, NULL, 0, NULL}        /* Sentinel */
};



void CUdmApp::UdmMain(Udm::DataNetwork* p_backend,
					 Udm::Object focusObject,
					 std::set<Udm::Object> selectedObjects,
					 long param, map<_bstr_t, _variant_t>& componentParameters, std::string workingDir)
{
	//AllocConsole();
	//Py_DebugFlag = 1;
	//Py_VerboseFlag = 2;
	Py_NoSiteFlag = 1; // we import site after setting up sys.path
	Py_Initialize();

	//PY_MAJOR_VERSION 
	//PY_MINOR_VERSION

	char *path = Py_GetPath();
#ifdef _WIN32
	std::string separator = ";";
#else
	std::string separator = ":";
#endif
	std::string newpath = path;
	std::string metapath;
	HKEY software_meta;
	if (RegOpenKeyExA(HKEY_LOCAL_MACHINE, "Software\\META", 0, KEY_READ, &software_meta) == ERROR_SUCCESS)
	{
		BYTE data[MAX_PATH];
		DWORD type, size = sizeof(data) / sizeof(data[0]);
		if (RegQueryValueExA(software_meta, "META_PATH", 0, &type, data, &size) == ERROR_SUCCESS)
		{
			metapath = std::string(data, data + strnlen((const char*)data, size));
		}
	}
	if (metapath.length())
	{
		newpath += separator + metapath + "\\bin";
	}
	//newpath += separator + "C:\\Program Files\\ISIS\\Udm\\bin";

	PySys_SetPath(const_cast<char*>(newpath.c_str()));

	PyObject_RAII main = PyImport_ImportModule("__main__");
	PyObject* main_namespace = PyModule_GetDict(main);
	PyObject_RAII ret = PyRun_StringFlags("import sys\n"
		"import udm\n", Py_file_input, main_namespace, main_namespace, NULL);
	if (ret == NULL && PyErr_Occurred())
	{
		throw python_error(GetPythonError());
	}

	HMODULE udm_pyd = LoadLibraryA("udm.pyd");
	if (!udm_pyd)
	{
		throw python_error("Could not load Udm Python: LoadLibary failed");
	}
	Object_Convert_t Object_Convert = (Object_Convert_t) GetProcAddress(udm_pyd, "Object_Convert");
	if (!Object_Convert)
	{
		FreeLibrary(udm_pyd);
		throw python_error("Could not load Udm Python: GetProcAddress failed");
	}
	PyObject_RAII pyFocusObject;
	if (focusObject)
		pyFocusObject.p = (*Object_Convert)(focusObject);
	else
	{
		pyFocusObject.p = Py_None;
		Py_INCREF(pyFocusObject.p);
	}
	PyObject_RAII pyRootObject = (*Object_Convert)(p_backend->GetRootObject());
	FreeLibrary(udm_pyd);
	
	Py_InitModule("CyPhyPython", CyPhyPython_methods);

	std::string module_name = _bstr_t(componentParameters.find(_bstr_t(L"script_file"))->second.bstrVal);
	if (module_name != "" && PathIsRelativeA(module_name.c_str())) {
		std::replace(module_name.begin(), module_name.end(), '/', '\\');
		if (module_name.rfind('\\') != std::string::npos)
		{
			std::string path = module_name.substr(0, module_name.rfind('\\'));

			newpath = workingDir + "\\" + path + separator + newpath;
			PySys_SetPath(const_cast<char*>(newpath.c_str()));
			module_name = module_name.substr(module_name.rfind('\\') + 1);
		}
		else
		{
			newpath = workingDir + separator + newpath;
			PySys_SetPath(const_cast<char*>(newpath.c_str()));
		}

		//newpath += separator + fullpath;
		//PySys_SetPath(const_cast<char*>(newpath.c_str()));
	} else {
		std::string scriptFilename;
		if (module_name != "") {
			scriptFilename = module_name;
		} else {
			scriptFilename = openfilename("Python Scripts (*.py)\0*.py\0");
		}
		if (scriptFilename.length() == 0)
		{
			return;
		}
		TCHAR fullpath[MAX_PATH];
		TCHAR* filepart;
		if (!GetFullPathNameA(scriptFilename.c_str(), sizeof(fullpath)/sizeof(fullpath[0]), fullpath, &filepart)) {
		} else {
			*(filepart-1) = '\0';

			newpath += separator + fullpath;
			PySys_SetPath(const_cast<char*>(newpath.c_str()));

			module_name = filepart;
		}
	}
	if (module_name.rfind(".py") != std::string::npos)
	{
		module_name = module_name.substr(0, module_name.rfind(".py"));
	}
	if (module_name != "")
	{
		PyObject_RAII module = PyImport_ImportModule(module_name.c_str());
		if (!module)
		{
			throw python_error(GetPythonError());
		}
		// TODO: check if this is necessary
		PyObject_RAII reloaded_module = PyImport_ReloadModule(module);
		if (!reloaded_module)
		{
			throw python_error(GetPythonError());
		}
		PyObject_RAII invoke = PyObject_GetAttrString(reloaded_module, "invoke");
		if (!invoke)
		{
			throw python_error("Error: script has no \"invoke\" function");
		}
		if (!PyCallable_Check(invoke))
		{
			throw python_error("Error: script \"invoke\" attribute is not callable");
		}

		PyObject_RAII empty_tuple = PyTuple_New(0);
		PyObject_RAII args = PyDict_New();
		PyDict_SetItemString(args, "focusObject", pyFocusObject);
		PyDict_SetItemString(args, "rootObject", pyRootObject);

		PyObject_RAII parameters = PyDict_New();
		for (auto it = componentParameters.begin(); it != componentParameters.end(); it++)
		{
			if (it->second.vt == VT_BSTR)
			{
				PyObject_RAII parameterString = PyUnicode_FromWideChar(it->second.bstrVal, SysStringLen(it->second.bstrVal));
				PyDict_SetItemString(parameters, static_cast<const char*>(it->first), parameterString);
			}
		}
		PyDict_SetItemString(args, "componentParameters", parameters);

		PyObject_RAII ret = PyObject_Call(invoke, empty_tuple, args);
		if (ret == NULL)
		{
			throw python_error(GetPythonError());
		}
		char* params[] = { "runCommand", "labels", NULL };
		for (char** param = params; *param; param++)
		{
			PyObject* pyParam = PyDict_GetItemString(parameters, *param);
			if (pyParam)
			{
				if (PyString_Check(pyParam))
				{
					componentParameters[_bstr_t(*param)] = _bstr_t(PyString_AsString(pyParam));
				}
				if (PyUnicode_Check(pyParam))
				{
					componentParameters[_bstr_t(*param)] = _bstr_t(PyUnicode_AsUnicode(pyParam));
				}
			}
		}
	}
}
