/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.IO;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Core;
using System.Linq;
using Microsoft.Win32;

using Common = ISIS.GME.Common;
using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using ISIS.GME.Common.Interfaces;
using System.Windows.Forms;




namespace CyPhySignalBlocksAddOn
{
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class CyPhySignalBlocksAddOnAddon : IMgaComponentEx, IGMEVersionInfo, IMgaEventSink
    {

        private MgaAddOn addon;
        private bool componentEnabled = true;
        private bool firstTime = false;
        private bool attachedSignalFlowLibrary = false;
        private bool attachedQudtLibrary = false;
        private string metaPath;

        private GMEConsole GMEConsole { get; set; }
        private MgaProject project;

        private LibraryTimer signalFlowTimer;
        private LibraryTimer qudtTimer;

        private delegate void TimerLogicDelegate();

        private class LibraryTimer
        {

            private string storeName;
            public string Name { get { return storeName; } }
            private bool setupTimer = false;
            private System.Windows.Forms.Timer timer = new System.Windows.Forms.Timer();
            private TimerLogicDelegate callback;
            private IMgaProject project;

            public LibraryTimer(string name, TimerLogicDelegate f, IMgaProject project)
            {
                callback = f;
                storeName = name;
                this.project = project;
            }

            public void TimerEventProcessor(Object obj, EventArgs evtArgs)
            {
                if ((project.ProjectStatus & 8) == 0) // if not in transaction
                {
                    timer.Stop();
                    callback();
                }
            }

            public void go()
            {
                if (!setupTimer)
                {
                    timer.Tick += new EventHandler(TimerEventProcessor);
                    timer.Interval = 1;
                    timer.Start();
                    setupTimer = true;
                }
                //else
                //{
                //timer.Enabled = true; // restart the timer -- unused right now because library refresh is not useful
                //}
            }
        }

        public CyPhySignalBlocksAddOnAddon()
        {

        }

        // Event handlers for addons
        #region MgaEventSink members
        public void GlobalEvent(globalevent_enum @event)
        {
            if (@event == globalevent_enum.GLOBALEVENT_CLOSE_PROJECT)
            {
                Marshal.FinalReleaseComObject(addon);
                addon = null;
            }
            if (!componentEnabled)
            {
                return;
            }
            if (@event == globalevent_enum.GLOBALEVENT_OPEN_PROJECT)
            {
                if (Environment.Is64BitProcess)
                {
                    MessageBox.Show("You're running GME (64 bit). Many CyPhyML components do not work with GME (64 bit). Please start GME instead.");
                }
            }
            if (@event == (globalevent_enum)11 /* GLOBALEVENT_OPEN_PROJECT_FINISHED */)
            {
                TriggerQudtRefreshIfNeeded();
            }

            // TODO: Handle global events
            // MessageBox.Show(@event.ToString());
        }

        // Look up the value of the META_PATH variable
        public string GetMetaPathValue()
        {
            const string keyName = "Software\\META";

            RegistryKey metaKey = RegistryKey.OpenBaseKey(RegistryHive.LocalMachine, RegistryView.Registry32)
                .OpenSubKey(keyName, false);

            string metaPath = "C:\\Program Files (x86)\\META";
            if (metaKey != null)
            {
                metaPath = (string)metaKey.GetValue("META_PATH", metaPath);
            }

            return metaPath;
        }


        /// <summary>
        /// Called when an FCO or folder changes
        /// </summary>
        /// <param name="subject">the object the event(s) happened to</param>
        /// <param name="eventMask">objectevent_enum values ORed together</param>
        /// <param name="param">extra information provided for cetertain event types</param>
        public void ObjectEvent(MgaObject subject, uint eventMask, object param)
        {
            // TODO: Handle object events (OR eventMask with the members of objectevent_enum)
            // Warning: Only those events are received that you have subscribed for by setting ComponentConfig.eventMask
            if (GMEConsole == null)
            {
                GMEConsole = GMEConsole.CreateFromProject(project);
            }
            TriggerQudtRefreshIfNeeded();
            if (enabled && subject.MetaBase.Name == "SignalFlowModel")
            {
                signalFlowTimer.go();
            }
        }

        private void TriggerQudtRefreshIfNeeded()
        {
            if (!firstTime)
            {
                // put the arguments into private variables where they can be reached by the timer code
                if (GMEConsole == null)
                {
                    GMEConsole = GMEConsole.CreateFromProject(project);
                }
#if DEBUG
                GMEConsole.Info.WriteLine("CyPhySignalBlocksAddOn starting up...");
#endif

                //GMEConsole.Info.WriteLine(eventMask.ToString());
                if (!componentEnabled)
                {
                    GMEConsole.Info.WriteLine("CyPhySignalBlocksAddOn not enabled...");
                    return;
                }

                // First, check to see whether the libraries have already been loaded
                //CyPhy.RootFolder rootFolder = ISIS.GME.Common.Utils.CreateObject<CyPhyClasses.RootFolder>(rf as MgaObject);

                IMgaFolder rootFolder = project.RootFolder;
                IMgaFolders subFolders = rootFolder.ChildFolders;
                foreach (IMgaFolder subFolder in subFolders)
                {
                    if (subFolder.Name.IndexOf("SignalFlowBlocks") != -1)
                    {
                        GMEConsole.Info.WriteLine("SignalFlow blocks library already loaded...");
                        attachedSignalFlowLibrary = true;
                    }
                }

                // Run this on any event, but only once (if not already loaded)
                if (!attachedQudtLibrary)
                {
                    qudtTimer.go();
                }

                firstTime = true;
            }
        }

        //private static void SignalFlowTimerEventProcessor( Object obj, EventArgs evtArgs )
        public void SignalFlowTimerHandler()
        {

            if (!attachedSignalFlowLibrary)
            {
                //GMEConsole.Info.WriteLine("Sending notification to attach SignalFlow library...");

                project.Notify(globalevent_enum.APPEVENT_LIB_ATTACH_BEGIN);
                try
                {
                    // GMEConsole.Info.WriteLine("Sent notification.");

                    string mgaPath = metaPath + "\\SignalFlowBlocks.mga";
                    GMEConsole.Info.WriteLine("Attaching library " + mgaPath);
                    project.BeginTransaction(project.ActiveTerritory);
                    RootFolder lib = Common.Classes.RootFolder.GetRootFolder(project).AttachLibrary("MGA=" + mgaPath);
                    project.CommitTransaction();
                }
                catch
                {
                    project.AbortTransaction();
                }
                finally
                {
                    project.Notify(globalevent_enum.APPEVENT_LIB_ATTACH_END);
                }
                GMEConsole.Info.WriteLine("Attached SignalFlowBlocks library.");
                attachedSignalFlowLibrary = true;
            }
            else
            {
                project.Notify(globalevent_enum.APPEVENT_LIB_ATTACH_BEGIN);
                try
                {
                    // KMS: FIXME probably needs to be Library.RefreshLibrary
                    string mgaPath = "MGA=" + metaPath + "\\SignalFlowBlocks.mga";
                    project.BeginTransaction(project.ActiveTerritory);
                    project.RootFolder.RefreshLibrary(mgaPath);
                    project.CommitTransaction();
                }
                catch
                {
                    project.AbortTransaction();
                }
                finally
                {

                    project.Notify(globalevent_enum.APPEVENT_LIB_ATTACH_END);
                }
            }
        }

        //private static void SignalFlowTimerEventProcessor( Object obj, EventArgs evtArgs )
        public void QudtTimerHandler()
        {

            string mgaPath = metaPath + "\\CyPhyMLQudt.mga";
            if (!attachedQudtLibrary)
            {
                if (!File.Exists(mgaPath))
                {
                    GMEConsole.Error.WriteLine("Path '" + mgaPath + "' does not exist. Cannot attach QUDT");
                    return;
                }
#if DEBUG
                //GMEConsole.Info.WriteLine("Sending notification to attach Qudt library...");
#endif
                project.Notify(globalevent_enum.APPEVENT_LIB_ATTACH_BEGIN);
                try
                {
                    //GMEConsole.Info.WriteLine("Sent notification.");

                    project.BeginTransaction(project.ActiveTerritory);

                    IMgaFolder oldQudt = project.RootFolder.ChildFolders.Cast<IMgaFolder>()
                        .Where(x => x.LibraryName != "" && (x.Name.Contains("Qudt") || x.Name.Contains("QUDT"))).FirstOrDefault();


                    bool needAttach;
                    if (oldQudt == null)
                    {
                        needAttach = true;
                    }
                    else
                    {
                        DateTime oldModTime;
                        long loldModTime;
                        if (long.TryParse(oldQudt.RegistryValue["modtime"], out loldModTime))
                        {
                            oldModTime = DateTime.FromFileTimeUtc(loldModTime);
                        }
                        else
                        {
                            oldModTime = DateTime.MinValue;
                        }
                        needAttach = File.GetLastWriteTimeUtc(mgaPath).CompareTo(oldModTime) > 0;
                        if (!needAttach)
                        {
                            GMEConsole.Info.WriteLine("QUDT is up-to-date: embedded library modified " + oldModTime.ToString() +
                                ", CyPhyMLQudt.mga modified " + File.GetLastWriteTimeUtc(mgaPath).ToString());
                        }
                    }
                    if (needAttach)
                    {
                        GMEConsole.Info.WriteLine("Attaching library " + mgaPath);
                        RootFolder newQudt = Common.Classes.RootFolder.GetRootFolder(project).AttachLibrary("MGA=" + mgaPath);
                        DateTime modtime = File.GetLastWriteTimeUtc(mgaPath);
                        ((newQudt as ISIS.GME.Common.Classes.RootFolder).Impl as GME.MGA.IMgaFolder).RegistryValue["modtime"] =
                             modtime.ToFileTimeUtc().ToString();

                        if (oldQudt != null)
                        {
                            ReferenceSwitcher.Switcher sw = new ReferenceSwitcher.Switcher(oldQudt, newQudt.Impl, null);
                            sw.UpdateSublibrary();
                            oldQudt.DestroyObject();
                        }
                        ((newQudt as ISIS.GME.Common.Classes.RootFolder).Impl as GME.MGA.IMgaFolder).LibraryName = "UnitLibrary QUDT";
                        GMEConsole.Info.WriteLine((oldQudt == null ? "Attached " : "Refreshed") + " Qudt library.");
                    }
                    project.CommitTransaction();
                }
                catch
                {
                    project.AbortTransaction();
                }
                finally
                {
                    project.Notify(globalevent_enum.APPEVENT_LIB_ATTACH_END);
                }
                attachedQudtLibrary = true;
            }
        }

        #endregion

        #region IMgaComponentEx Members


        public void Initialize(MgaProject p)
        {
            // Creating addon
            p.CreateAddOn(this, out addon);
            // Setting event mask (see ComponentConfig.eventMask)
            unchecked
            {
                addon.EventMask = (uint)ComponentConfig.eventMask;
            }
            this.project = p;
            if (metaPath == null)
            {
                metaPath = Path.Combine(GetMetaPathValue(), "meta");
            }

            if (!Directory.Exists(metaPath))
            {
                throw new ApplicationException(metaPath + " doesn't exist");
            }
            signalFlowTimer = new LibraryTimer("SignalFlowTimer", new TimerLogicDelegate(SignalFlowTimerHandler), project);
            qudtTimer = new LibraryTimer("QudtTimer", new TimerLogicDelegate(QudtTimerHandler), project);
        }

        public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException(); // Not called by addon
        }


        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            object value;
            if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            if (componentParameters == null)
            {
                componentParameters = new SortedDictionary<string, object>();
            }

            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();
        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion
    }
}
