/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Core;
using System.Linq;
using System.Diagnostics;

namespace CyPhySoT
{
    /// <summary>
    /// This class implements the necessary COM interfaces for a GME interpreter component.
    /// </summary>
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class CyPhySoTInterpreter : IMgaComponentEx, IGMEVersionInfo
    {
        [ComVisible(false)]
        public const string RunCommand = "";
        [ComVisible(false)]
        public string OutputBaseDir { get; set; }
        [ComVisible(false)]
        public string ProjectFilename { get; set; }

        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        public enum ComponentStartMode
        {
            GME_MAIN_START = 0, 		// Not used by GME
            GME_BROWSER_START = 1,      // Right click in the GME Tree Browser window
            GME_CONTEXT_START = 2,		// Using the context menu by right clicking a model element in the GME modeling window
            GME_EMBEDDED_START = 3,		// Not used by GME
            GME_MENU_START = 16,		// Clicking on the toolbar icon, or using the main menu
            GME_BGCONTEXT_START = 18,	// Using the context menu by right clicking the background of the GME modeling window
            GME_ICON_START = 32,		// Not used by GME
            GME_SILENT_MODE = 128 		// Not used by GME, available to testers not using GME
        }

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a tansaction.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        public void Initialize(MgaProject project)
        {
            // set up the default component parameters
            this.componentParameters = new SortedDictionary<string, object>();
            this.componentParameters.Add("labels", "");
            this.componentParameters.Add("runCommand", RunCommand);
            this.componentParameters.Add("output_dir", string.Empty);
            this.componentParameters.Add("automation", "false");
            this.componentParameters.Add("run_silent", "false");
            this.componentParameters.Add("do_config", "true");
            //this.componentParameters.Add("configuration", string.Empty);
            //this.componentParameters.Add("TestBench", string.Empty);
            //this.componentParameters.Add("TestBenchUniqueName", string.Empty);
            //this.componentParameters.Add("test_bench_name", string.Empty);
        }

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is available. A general try-catch block catches all the exceptions
        /// coming from this function, you don't need to add it. For more information, see InvokeEx.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        /// <param name="currentobj">The model open in the active tab in GME. Its value is null if no model is open (no GME modeling windows open). </param>
        /// <param name="selectedobjs">
        /// A collection for the selected model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the context menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the collection is empty (contains zero elements).
        /// </param>
        /// <param name="startMode">Contains information about the GUI event that initiated the invocation.</param>
        [ComVisible(false)]
        public void Main(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, ComponentStartMode startMode)
        {
            GMEConsole.Info.WriteLine("Running {0} interpreter...", ComponentName);

            this.ProjectFilename = project.ProjectConnStr.Substring("MGA=".Length);

            var error = CheckModel(currentobj);
            if (error)
            {
                GMEConsole.Error.WriteLine("At least one critical error occured in the model. No out files were generated.");
                return;
            }

            var projectDir = Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length));
            var outputDir = Path.Combine(projectDir, "results", currentobj.Name);

            if (string.IsNullOrWhiteSpace(this.componentParameters["output_dir"] as string))
            {
                this.componentParameters["output_dir"] = outputDir;
            }
            else
            {
                outputDir = this.componentParameters["output_dir"] as string;
            }

            if (Directory.Exists(outputDir) == false)
            {
                Directory.CreateDirectory(outputDir);
            }

            this.OutputBaseDir = outputDir;

            this.Process(currentobj);

            GMEConsole.Info.WriteLine("{0} interpreter is done", ComponentName);
        }

        private bool CheckModel(MgaFCO currentobj)
        {
            bool error = false;

            // checks

            // current object cannot be null
            if (currentobj == null)
            {
                GMEConsole.Error.WriteLine("Please open a TestBenchSuite model.");
                error = true;
                return error;
            }

            // current object must be a Test bench suite
            if (currentobj.Meta.Name != "TestBenchSuite")
            {
                GMEConsole.Error.WriteLine("Please open a TestBenchSuite model.");
                error = true;
                return error;
            }

            // check for unsupported object types
            UnsupportedKinds = new List<string>();
            UnsupportedKinds.AddRange(new string[] { "SystemUnderTest" });

            var fcos = (currentobj as MgaModel)
                .ChildObjects
                .Cast<MgaFCO>()
                .Where(x => UnsupportedKinds.Contains(x.Meta.Name));

            foreach (var item in fcos)
            {
                GMEConsole.Error.WriteLine("{0}[{1}] is not supported {2}",
                    item.Name,
                    item.Meta.Name,
                    item.ToMgaHyperLink());

                error = true;
            }

            // no TB null ref
            var tbRefs = (currentobj as MgaModel)
                .ChildObjects
                .OfType<MgaReference>()
                .Where(x => x.Meta.Name == "TestBenchRef");


            var tbs = tbRefs.Where(x => x.Referred != null).Select(x => x.Referred);

            if (tbs.Count() != tbRefs.Count())
            {
                error = true;
                GMEConsole.Error.WriteLine("Some test benches are null references.");
                return error;
            }

            List<MgaFCO> uniqueTBs = new List<MgaFCO>();
            // one test bench can be contained only once.
            foreach (var item in tbs)
            {
                if (uniqueTBs.FirstOrDefault(x => x.ID == item.ID) == null)
                {
                    uniqueTBs.Add(item);
                }
                else
                {
                    error = true;
                    GMEConsole.Error.WriteLine("One test bench can be contained only once. {0}", item.ToMgaHyperLink());
                    return error;
                }
            }

            MgaFCO referredRootDesignNode = null;

            foreach (var tb in tbs)
            {
                var suts = tb.ChildObjects.Cast<IMgaFCO>().Where(x => x.MetaBase.Name == "TopLevelSystemUnderTest");

                if (suts.Count() != 1)
                {
                    // referenced test benches have one system under test object
                    error = true;
                    string msg = string.Format("Test bench must have one system under test object. {0}",
                        tb.ToMgaHyperLink());

                    GMEConsole.Error.WriteLine(msg);
                    Trace.TraceError(msg);
                }
                else
                {
                    if ((suts.FirstOrDefault() as MgaReference).Referred.RootFCO.MetaBase.Name == "DesignContainer")
                    {
                        error = true;
                        // cannot point to a design space
                        GMEConsole.Error.WriteLine(
                            "Test bench's system under test object: {0} points to a design space {1}. Use master interpreter instead.",
                            tb.ToMgaHyperLink(),
                            (suts.FirstOrDefault() as MgaReference).Referred.RootFCO.ToMgaHyperLink());
                    }

                    // referenced test bechnes have system under test object referred to the same design
                    if (referredRootDesignNode == null)
                    {
                        referredRootDesignNode = (suts.FirstOrDefault() as MgaReference).Referred.RootFCO;
                    }
                    else if (referredRootDesignNode.ID != (suts.FirstOrDefault() as MgaReference).Referred.RootFCO.ID)
                    {
                        error = true;
                        GMEConsole.Error.WriteLine(
                            "Test bench: {0} refers to {1} design, while other test benches refer to: {2}",
                            tb.ToMgaHyperLink(),
                            (suts.FirstOrDefault() as MgaReference).Referred.RootFCO.ToMgaHyperLink(),
                            referredRootDesignNode.ToMgaHyperLink());
                    }
                }

                // TODO: must have workflow with one task in it
                var workflowRefs = tb
                    .ChildObjects
                    .OfType<MgaReference>()
                    .Where(x => x.Meta.Name == "WorkflowRef");

                if (workflowRefs.Count() != 1)
                {
                    GMEConsole.Error.WriteLine("There must be exactly one workflow reference in {0}", tb.ToMgaHyperLink());
                    error = true;
                }

                var workflowRef = workflowRefs.FirstOrDefault();

                if (workflowRef != null)
                {
                    if (workflowRef.Referred == null)
                    {
                        GMEConsole.Error.WriteLine("Workflow reference cannot be null {0}", workflowRef.Name);
                        error = true;
                    }
                    else
                    {
                        var tasks = workflowRef.Referred.ChildObjects;
                        if (tasks.Count == 1)
                        {
                            var task = tasks.Cast<MgaFCO>().FirstOrDefault();
                            if (string.IsNullOrWhiteSpace(task.StrAttrByName["COMName"]))
                            {
                                GMEConsole.Error.WriteLine("COMName cannot be an empty string for {0}", task.Name);
                                error = true;
                            }
                        }
                        else
                        {
                            GMEConsole.Error.WriteLine("There must be exactly one task in {0}", workflowRef.Referred.Name);
                            error = true;
                        }
                    }
                }

            }

            // unique TB names
            var names = tbRefs.Select(x => x.Name).ToList();
            if (names.Distinct().Count() < names.Count())
            {
                GMEConsole.Error.WriteLine("Test bench references must have unique names");
                names.Sort();
                names.ForEach(x => GMEConsole.Info.WriteLine("{0}", x));
                error = true;
            }

            error = UpdateDependency((currentobj as MgaModel));

            // checks are done
            return error;
        }

        class CycleException : Exception
        {
            public CycleException(string message)
                : base(message)
            {
            }
        }

        public List<List<MgaFCO>> DependencyGraph { get; set; }

        delegate void dfs_visit(MgaFCO fco, ref List<MgaFCO> out_);
        /// <summary>
        /// 
        /// </summary>
        /// <returns>True if dependency cannot be determined.</returns>
        private bool UpdateDependency(MgaModel testBenchSuite)
        {
            GMEConsole.Info.WriteLine("Updating test bench dependencies ...");

            int unprocessedFCOs = 0;
            HashSet<MgaFCO> toProcess = new HashSet<MgaFCO>(new MgaObjectEqualityComparor<MgaFCO>());
            // Find all FCOs with no outgoing connections
            foreach (MgaFCO fco in testBenchSuite.ChildFCOs.Cast<MgaFCO>().Where(fco => fco.ObjType != GME.MGA.Meta.objtype_enum.OBJTYPE_CONNECTION))
            {
                unprocessedFCOs++;
                if (fco.PartOfConns.Cast<MgaConnPoint>().Where(connpoint => connpoint.ConnRole == "src" && connpoint.References.Count == 0).Count() == 0)
                {
                    if (fco.ObjType == GME.MGA.Meta.objtype_enum.OBJTYPE_REFERENCE)
                    {
                        MgaReference reference = (MgaReference)fco;
                        if (reference.UsedByConns.Cast<MgaConnPoint>().Where(connpoint => connpoint.ConnRole == "src").Count() == 0)
                        {
                            toProcess.Add(fco);
                        }
                    }
                    else
                    {
                        toProcess.Add(fco);
                    }
                }
            }

            this.DependencyGraph = new List<List<MgaFCO>>();

            HashSet<MgaFCO> tempMarked = new HashSet<MgaFCO>(new MgaObjectEqualityComparor<MgaFCO>());
            Dictionary<MgaFCO, List<MgaFCO>> resident = new Dictionary<MgaFCO, List<MgaFCO>>(new MgaObjectEqualityComparor<MgaFCO>());

            dfs_visit Visit = null;
            Visit = delegate(MgaFCO fco, ref List<MgaFCO> out_)
            {
                if (tempMarked.Contains(fco))
                {
                    throw new CycleException("Cycle involving " + fco.Name);
                }
                List<MgaFCO> previousOut;
                if (!resident.TryGetValue(fco, out previousOut))
                {
                    //GMEConsole.Out.WriteLine("" + fco.Name);
                    tempMarked.Add(fco);
                    IEnumerable<MgaConnPoint> connections = fco.PartOfConns.Cast<MgaConnPoint>().Where(cp => cp.ConnRole != "src");
                    if (fco.ObjType == GME.MGA.Meta.objtype_enum.OBJTYPE_REFERENCE)
                    {
                        MgaReference reference = (MgaReference)fco;
                        connections = connections.Concat(reference.UsedByConns.Cast<MgaConnPoint>().Where(cp => cp.ConnRole != "src" && cp.References[1] == reference));
                    }
                    foreach (MgaConnPoint connpoint in connections)
                    {
                        MgaSimpleConnection conn = (MgaSimpleConnection)connpoint.Owner;
                        MgaFCO other = conn.Src;
                        if (conn.SrcReferences.Count > 0)
                        {
                            other = (connpoint.Owner as MgaSimpleConnection).SrcReferences[1];
                        }
                        // GMEConsole.Out.WriteLine("&nbsp; " + other.Name);
                        Visit(other, ref out_);
                    }
                    tempMarked.Remove(fco);
                    out_.Add(fco);
                    resident.Add(fco, out_);
                    unprocessedFCOs--;
                }
                else
                {
                    if (previousOut != out_)
                    {
                        this.DependencyGraph.Remove(out_);
                        foreach (MgaFCO fco2 in out_)
                        {
                            resident[fco2] = previousOut;
                        }
                        previousOut.AddRange(out_);
                        out_ = previousOut;
                    }
                }
            };

            while (toProcess.Count > 0)
            {
                var before = toProcess.Count;

                List<MgaFCO> out_ = new List<MgaFCO>();
                this.DependencyGraph.Add(out_);
                MgaFCO fco = toProcess.First();
                try
                {
                    Visit(fco, ref out_);
                }
                catch (CycleException ex)
                {
                    GMEConsole.Error.WriteLine(ex.Message);
                    Trace.TraceError(ex.ToString());
                    return true;
                }

                foreach (MgaFCO fco2 in out_)
                {
                    toProcess.Remove(fco2);
                }

                var after = toProcess.Count;
                if (after >= before)
                {
                    string msg = "Determining dependency is not progressing.";
                    GMEConsole.Error.WriteLine(msg);
                    Trace.TraceError(msg);
                    return true;
                }
            }

            int i = 0;
            foreach (List<MgaFCO> fcos in this.DependencyGraph)
            {
                GMEConsole.Info.WriteLine("Dependency chain [{0}]: {1}", i++, string.Join(" ", fcos.Select(x => x.Name)));
            }

            if (unprocessedFCOs != 0)
            {
                List<MgaFCO> allFCOs = testBenchSuite.ChildFCOs.Cast<MgaFCO>().Where(fco => fco.ObjType != GME.MGA.Meta.objtype_enum.OBJTYPE_CONNECTION).ToList();
                foreach (List<MgaFCO> fcos in this.DependencyGraph)
                {
                    foreach (MgaFCO fco in fcos)
                    {
                        allFCOs.Remove(fco);
                    }
                }
                GMEConsole.Error.WriteLine("Cycle involving " + string.Join(" ", allFCOs.Select(x => x.Name)));
                return true;
            }

            return false;
        }


        private SotConfig sotConfig = new SotConfig();

        public List<TestBench> TestBenches = new List<TestBench>();


        private void Process(MgaFCO currentobj)
        {
            sotConfig.SoTID = currentobj.ID;

            var tbRefs = (currentobj as MgaModel)
                .ChildObjects
                .OfType<MgaReference>()
                .Where(x => x.Meta.Name == "TestBenchRef");

            var outputBaseDir = this.componentParameters["output_dir"] as string;

            foreach (var tbRef in tbRefs)
            {
                var task = tbRef
                    .Referred
                    .ChildObjects
                    .OfType<MgaReference>()
                    .FirstOrDefault(x => x.Meta.Name == "WorkflowRef")
                    .Referred
                    .ChildObjects
                    .OfType<MgaAtom>()
                    .FirstOrDefault();

                var parameters = task.StrAttrByName["Parameters"];
                var parametersDict = new Dictionary<string, string>();
                if (string.IsNullOrWhiteSpace(parameters) == false)
                {
                    try
                    {
                        parametersDict = Newtonsoft.Json.JsonConvert.DeserializeObject<Dictionary<string, string>>(parameters);
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(ex);
                    }
                }

                var testbench = new TestBench();
                testbench.OutputDirectory = GetRandomTBDir(outputBaseDir);
                testbench.Project = tbRef.Project;
                testbench.CurrentObj = tbRef.Referred;
                testbench.ParamInvoke = paramInvoked;

                testbench.Name = string.Format("{0}__{1}", tbRef.Referred.RegistryValue["TestBenchUniqueName"], tbRef.Referred.Name);
                testbench.ProgId = task.StrAttrByName["COMName"];
                testbench.ParametersDict = parametersDict;

                TestBenches.Add(testbench);
            }

            if (this.DependencyGraph == null)
            {
                GMEConsole.Warning.WriteLine("Dependency was not determined running all test benches without dependencies.");
                return;
            }

            // Update Testbench upstream and downstream dependencies based on the MgaFCO dependency map/list
            foreach (var chain in this.DependencyGraph)
            {
                for (int i = 0; i < chain.Count; i++)
                {
                    TestBench beforeTb = null;
                    var currentTb = this.TestBenches.FirstOrDefault(x => x.CurrentObj == (chain[i] as MgaReference).Referred);
                    TestBench afterTb = null;
                    if (currentTb == null)
                    {
                        continue;
                    }
                    if (i > 0)
                    {
                        beforeTb = this.TestBenches.FirstOrDefault(x => x.CurrentObj == (chain[i - 1] as MgaReference).Referred);
                    }
                    if (i < chain.Count - 1)
                    {
                        afterTb = this.TestBenches.FirstOrDefault(x => x.CurrentObj == (chain[i + 1] as MgaReference).Referred);
                    }

                    if (beforeTb != null)
                    {
                        currentTb.UpstreamTestBenches.Add(beforeTb);
                    }

                    if (afterTb != null)
                    {
                        currentTb.DownstreamTestBenches.Add(afterTb);
                    }
                }
            }
        }


        private string GetRandomTBDir(string outputBaseDir)
        {
            string randomFolderName = Path.GetFileNameWithoutExtension(Path.GetRandomFileName());
            List<char> illegalStartChars = new List<char>()
                    {
                        '0','1','2','3','4','5','6','7','8','9'
                    };

            string OutputSubDir = Path.Combine(
                outputBaseDir,
                randomFolderName);

            int maxFolders = 0;

            while (illegalStartChars.Contains(randomFolderName.FirstOrDefault()) ||
                File.Exists(OutputSubDir) ||
                Directory.Exists(OutputSubDir))
            {
                if (maxFolders++ > 2000000)
                {
                    throw new Exception(
                        string.Format("Number of tries ({0}) to create an output folder exceeded in {0}. ",
                            maxFolders,
                            outputBaseDir));
                }

                randomFolderName = Path.GetFileNameWithoutExtension(Path.GetRandomFileName());

                OutputSubDir = Path.Combine(
                    outputBaseDir,
                    randomFolderName);
            }

            if (this.componentParameters["run_silent"] as string == "true")
            {
                if (Directory.Exists(OutputSubDir) == false)
                {
                    Directory.CreateDirectory(OutputSubDir);
                }
            }

            return OutputSubDir;
        }


        private void CallTestBenchInterpreters()
        {
            // this function must be outside of any transaction

            // TODO: order testbenches based on dependency (value flow)
            foreach (var testbench in TestBenches)
            {
                if (this.componentParameters["run_silent"] as string == "true")
                {
                    // TODO: Interpreter must run
                    GMEConsole.Info.WriteLine(
                        "Call interpreter in a silent mode with config setup (no dialog boxes): {0}",
                        testbench.ProgId);

                    testbench.ParametersDict["do_config"] = this.componentParameters["do_config"] as string;
                    testbench.ParametersDict["console_messages"] = "off";

                    // JobManagerLib will take care of the run. Just expose the testbench list.
                }
                else if (this.componentParameters["do_config"] as string == "true")
                {
                    // Interpreter must not run!
                    GMEConsole.Info.WriteLine("Using default config for interpreter: {0}", testbench.ProgId);
                }

                sotConfig.OriginalProjectFileName = this.ProjectFilename;
                //sotConfig.CommandsToRun.Add(testbench.Name, testbench.RunCommand);
            }

            if (this.componentParameters["run_silent"] as string == "false")
            {
                using (StreamWriter writer = new StreamWriter(Path.Combine(OutputBaseDir, "manifest.sot.json")))
                {
                    writer.WriteLine(Newtonsoft.Json.JsonConvert.SerializeObject(sotConfig, Newtonsoft.Json.Formatting.Indented));
                }
            }
        }


        internal int paramInvoked = 0;
        private List<string> UnsupportedKinds { get; set; }

        #region IMgaComponentEx Members

        MgaGateway MgaGateway { get; set; }
        GMEConsole GMEConsole { get; set; }

        string logFileName { get; set; }

        public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            paramInvoked = param;

            if (!enabled)
            {
                return;
            }

            string currentWorkDir = System.IO.Directory.GetCurrentDirectory();

            try
            {
                System.IO.Directory.SetCurrentDirectory(
                    Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length)));

                // logfile in the project directory
                this.logFileName = Path.Combine(
                    Environment.CurrentDirectory,
                    "log",
                    this.ComponentName + "." + System.Diagnostics.Process.GetCurrentProcess().Id + ".log");

                META.Logger.AddFileListener(this.logFileName, this.ComponentName);

                Trace.TraceInformation("InvokeEx called {0}", this.ComponentName);

                GMEConsole = GMEConsole.CreateFromProject(project);
                MgaGateway = new MgaGateway(project);
                project.CreateTerritoryWithoutSink(out MgaGateway.territory);

                GMEConsole.Info.WriteLine("Running {0}", this.ComponentName);

                MgaGateway.PerformInTransaction(delegate
                {
                    Main(project, currentobj, selectedobjs, Convert(param));
                });

                if (this.componentParameters["run_silent"] as string == "false")
                {
                    // Copy mga project file to output directory
                    string projectFile = project.ProjectConnStr.Substring("MGA=".Length);
                    string outputFile = Path.Combine(OutputBaseDir, Path.GetFileName(projectFile));
                    project.Save("MGA=" + outputFile, true);
                    sotConfig.ProjectFileName = Path.GetFileName(outputFile);

                    // CyPhy2CAD stores its config in an xml file instead of in the project
                    string CADConfigXml = Path.Combine(Path.GetDirectoryName(projectFile), "CyPhy2CAD_config.xml");
                    if (File.Exists(CADConfigXml))
                    {
                        string CADConfigXmlCopy = Path.Combine(OutputBaseDir, "CyPhy2CAD_config.xml");
                        File.Copy(CADConfigXml, CADConfigXmlCopy, true);
                    }

                    string ModelicaConfigXml = Path.Combine(Path.GetDirectoryName(projectFile),
                        CyPhy2Modelica.CyPhy2ModelicaSettings.ConfigFilename);

                    if (File.Exists(ModelicaConfigXml))
                    {
                        string ModelicaConfigXmlCopy = Path.Combine(OutputBaseDir,
                            CyPhy2Modelica.CyPhy2ModelicaSettings.ConfigFilename);

                        File.Copy(ModelicaConfigXml, ModelicaConfigXmlCopy, true);
                    }

                    // Copy PostProcessing scripts.
                    // TODO: fix this because it does not work
                    MgaGateway.PerformInTransaction(delegate
                    {
                        var postProcs = currentobj
                            .ChildObjects
                            .Cast<MgaFCO>()
                            .Where(x => x.MetaBase.Name == "PostProcessing");

                        foreach (var postProc in postProcs)
                        {
                            if (string.IsNullOrWhiteSpace(postProc.StrAttrByName["ScriptPath"]) == false &&
                               File.Exists(postProc.StrAttrByName["ScriptPath"]))
                            {
                                // Copy postScript file
                                FileInfo postScript = new FileInfo(Path.GetFullPath(postProc.StrAttrByName["ScriptPath"]));
                                string destFile = Path.Combine(OutputBaseDir, postScript.Name);
                                File.Copy(postScript.FullName, destFile, true);

                                // If common folder exist copy entire folder.
                                DirectoryInfo commonSrc = new DirectoryInfo(Path.Combine(postScript.DirectoryName, "common"));
                                DirectoryInfo commonTarget = new DirectoryInfo(Path.Combine(OutputBaseDir, "common"));
                                if (commonSrc.Exists && commonTarget.Exists == false)
                                {
                                    commonTarget.Create();
                                    CopyFilesRecursively(commonSrc, commonTarget);
                                }
                            }
                        }
                    });
                }

                CallTestBenchInterpreters();

                GMEConsole.Info.WriteLine("{0} has finished.", this.ComponentName);
            }
            catch (Exception ex)
            {
                Trace.TraceError(ex.ToString());
                GMEConsole.Error.WriteLine("{0} has finished with critical errors see log file.", this.ComponentName);
            }
            finally
            {
                if (MgaGateway.territory != null)
                {
                    MgaGateway.territory.Destroy();
                }
                MgaGateway = null;
                project = null;
                currentobj = null;
                selectedobjs = null;
                GMEConsole = null;
                GC.Collect();
                GC.WaitForPendingFinalizers();
                GC.Collect();

                System.IO.Directory.SetCurrentDirectory(currentWorkDir);
            }

            META.Logger.RemoveFileListener(this.ComponentName);
        }

        /// <summary>
        /// Helper function for copying entire common folder for post processing
        /// </summary>
        /// <param name="source">Source directory</param>
        /// <param name="target">Destination directory</param>
        private static void CopyFilesRecursively(
            DirectoryInfo source,
            DirectoryInfo target)
        {
            foreach (DirectoryInfo dir in source.GetDirectories())
            {
                CopyFilesRecursively(dir, target.CreateSubdirectory(dir.Name));
            }

            foreach (FileInfo file in source.GetFiles())
            {
                file.CopyTo(Path.Combine(target.FullName, file.Name), true);
            }
        }

        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            object value;
            if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            if (componentParameters == null)
            {
                componentParameters = new SortedDictionary<string, object>();
            }

            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();

        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion

    }

    class MgaObjectEqualityComparor<T> : EqualityComparer<T> where T : IMgaObject
    {
        public override bool Equals(T x, T y)
        {
            return x.ID.Equals(y.ID);
        }

        public override int GetHashCode(T obj)
        {
            return obj.ID.GetHashCode();
        }
    }
}
