/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Linq;
using System.IO;
using System.IO.Packaging;
using System.Runtime.Remoting.Channels;
using System.Net;
using System.Diagnostics;
using System.Threading;
using System.Web;
using System.Security;
using System.Data.SQLite;
using System.Windows.Forms.Design;

namespace JobManager
{
    public partial class JobManager : Form
    {
        private SpringLabel toolStripStatusLabel { get; set; }

        private RemoteServiceStatusForm remoteServiceStatusForm { get; set; }

        public NotifyIcon NotifyIcon;
        internal JobServer server;

        private string TempDir { get; set; }

        private enum Headers
        {
            Id, Title, Status, Progress, RunCommand, WorkingDirectory
        }

        private Dictionary<Job.TypeEnum, TargetMachine> runtimeConfig =
            new Dictionary<Job.TypeEnum, TargetMachine>()
		{
			{ Job.TypeEnum.Command,
				new TargetMachine(
					new Uri("tcp://localhost:35010/JobServer"),
					TargetMachine.TargetMachineType.Local)},

			{ Job.TypeEnum.Matlab,
				new TargetMachine(
					new Uri("tcp://localhost:35010/JobServer"),
					TargetMachine.TargetMachineType.Local)},

			{ Job.TypeEnum.CAD,
			  new TargetMachine(
			    new Uri("tcp://localhost:35010/JobServer"),
			    TargetMachine.TargetMachineType.Local)},
		};

        public class TargetMachine
        {
            public enum TargetMachineType
            {
                Local,
                Remote,
            }

            public Uri Connection { get; set; }
            public TargetMachineType Type { get; set; }

            public TargetMachine(
                Uri connection,
                TargetMachineType type = TargetMachineType.Local)
            {
                Connection = connection;
                Type = type;
            }
        }

        public void UpdateRuntimeConfig(Dictionary<Job.TypeEnum, TargetMachine> config)
        {
            foreach (var item in config)
            {
                if (runtimeConfig.ContainsKey(item.Key))
                {
                    runtimeConfig[item.Key] = item.Value;
                }
                else
                {
                    runtimeConfig.Add(item.Key, item.Value);
                }
            }
        }

        /// <summary>
        /// Local thread pool for jobs.
        /// </summary>
        private LocalPool pool = new LocalPool();

        /// <summary>
        /// Controls will be refreshed when the timer has elapsed.
        /// </summary>
        private System.Windows.Forms.Timer timer = new System.Windows.Forms.Timer()
        {
            Interval = 1000,
        };

        /// <summary>
        /// Checks if VF and remote executors are online or not.
        /// </summary>
        /// <returns>True if jobs can be posted, otherwise false.</returns>
        private bool IsServiceOnline()
        {
            // poll VF status
            bool serverOnline = false;
            bool jenkinsOnline = false;

            serverOnline = this.Jenkins.PingVF();
            jenkinsOnline = this.Jenkins.PingJenkins();

            return serverOnline && jenkinsOnline;
        }

        /// <summary>
        /// Updates all controls that displays status of the remote servers.
        /// </summary>
        private void UpdateServiceStatus()
        {
            Trace.TraceInformation("Pinging VF and getting status.");

            StringBuilder sb = new StringBuilder();

            global::JobManager.Jenkins.UserProfile.UserProfile info = null;
            if (this.Jenkins.PingVF(out info))
            {
                sb.AppendFormat("VF is connected {0} and logged in as {1}. ",
                    this.server.JenkinsUrl,
                    this.server.UserName);

                this.remoteServiceStatusForm.SafeInvoke(delegate
                {
                    this.remoteServiceStatusForm.lblVFStatus.Text = "OK";
                });

                global::JobManager.Jenkins.StatusInfo.StatusInfo statusInfo = null;

                if (this.Jenkins.PingJenkins(out statusInfo))
                {
                    // TODO: update detailed status
                    sb.AppendFormat(
                        "Remote job server is connected. [{0}: {1} available and {2} busy of {3} total executors]",
                        statusInfo.status,
                        statusInfo.totalAvailable,
                        statusInfo.totalBusy,
                        statusInfo.totalAvailable + statusInfo.totalBusy);

                    this.remoteServiceStatusForm.SafeInvoke(delegate
                    {
                        this.remoteServiceStatusForm.lblNodeStatus.Text = statusInfo.status;

                        this.remoteServiceStatusForm.lblBusyCount.Text = statusInfo.totalBusy.ToString();
                        this.remoteServiceStatusForm.lblFreeCount.Text = statusInfo.totalAvailable.ToString();

                        this.remoteServiceStatusForm.lblTotalCount.Text =
                            (statusInfo.totalAvailable + statusInfo.totalBusy).ToString();

                        this.remoteServiceStatusForm.lvRemoteNodes.SafeInvoke(delegate
                        {
                            this.remoteServiceStatusForm.lvRemoteNodes.Items.Clear();
                            foreach (var node in statusInfo.nodes)
                            {
                                var listViewItem = new ListViewItem(new string[] 
                                                    {
                                                        node.status,
                                                        node.busy.ToString(),
                                                        node.available.ToString(),
                                                        (node.available + node.busy).ToString(),
                                                        node.name,
                                                        node.description
                                                    });
                                this.remoteServiceStatusForm.lvRemoteNodes.Items.Add(listViewItem);
                            }
                        });

                        this.remoteServiceStatusForm.gbRemoteNodes.Visible = true;
                    });
                }
                else
                {
                    if (statusInfo != null)
                    {
                        sb.AppendFormat("Remote job server is NOT connected. [{0}]", statusInfo.status);

                        this.remoteServiceStatusForm.SafeInvoke(delegate
                        {
                            this.remoteServiceStatusForm.lblNodeStatus.Text = statusInfo.status;
                            this.remoteServiceStatusForm.gbRemoteNodes.Visible = false;
                        });
                    }
                    else
                    {
                        sb.AppendFormat("Remote job server is NOT connected.");

                        this.remoteServiceStatusForm.SafeInvoke(delegate
                        {
                            this.remoteServiceStatusForm.lblNodeStatus.Text = "DOWN";
                            this.remoteServiceStatusForm.gbRemoteNodes.Visible = false;
                        });

                    }
                }
            }
            else
            {
                sb.AppendFormat("VF is NOT connected. {0}", this.server.JenkinsUrl);
                this.remoteServiceStatusForm.SafeInvoke(delegate
                {
                    this.remoteServiceStatusForm.lblVFStatus.Text = "DOWN";
                    this.remoteServiceStatusForm.lblNodeStatus.Text = "DOWN";
                    this.remoteServiceStatusForm.gbRemoteNodes.Visible = false;
                });
            }

            this.toolStripStatusLabel.Text = sb.ToString();
        }

        private void MonitorJenkinsJobs(object parameter)
        {
            try
            {
                try
                {
                    this.UpdateServiceStatus();
                }
                catch (Exception ex)
                {
                    Trace.TraceError(ex.ToString());
                }

                foreach (var kvp2 in JobMap.ToList())
                {
                    var jenkinsJob = kvp2.Key;
                    var job = kvp2.Value;

                    try
                    {
                        if (this.IsServiceOnline())
                        {
                            try
                            {
                                // if online get updated job status
                                MonitorJenkinsJob(jenkinsJob, job);
                            }
                            catch (Exception ex)
                            {
                                job.Status = Job.StatusEnum.Failed;
                                Jenkins.DebugLog.WriteLine(job.Title + " failed: " + ex.ToString().Replace("\n", "\t"));
                            }
                        }
                        else
                        {
                            // if not online just skip
                            continue;
                        }
                    }
                    catch (Exception ex)
                    {
                        Trace.TraceError(ex.ToString());
                    }

                    if (job.IsFailed() || job.Status == Job.StatusEnum.Succeeded)
                    {
                        var jobToRemove = JobMap.FirstOrDefault(x => x.Key.url == jenkinsJob.url).Key;
                        if (jobToRemove != null)
                        {
                            JobMap.Remove(jobToRemove);
                        }
                    }
                }

                try
                {
                    this.UpdateServiceStatus();
                }
                catch (Exception ex)
                {
                    Trace.TraceError(ex.ToString());
                }
            }
            catch (Exception ex)
            {
                Trace.TraceWarning(ex.ToString());
            }

            // restarting timer is important nothing can fail above.
            this.RestartMonitorTimer();
        }

        private void RestartMonitorTimer(int dueTime = global::JobManager.Jenkins.Jenkins.VF_JOB_POLL_FREQUENCY)
        {
            Trace.TraceInformation("Restarting MonitorTimer for remote jobs due time: {0}.", dueTime);

            JenkinsTimer = new System.Threading.Timer(
                MonitorJenkinsJobs,
                null,
                dueTime,
                Timeout.Infinite);
        }

        private void MonitorJenkinsJob(global::JobManager.Jenkins.Job.Job jenkinsJob, Job job)
        {
            jenkinsJob = Jenkins.GetJobInfo(jenkinsJob.name);
            if (jenkinsJob == null)
            {
                job.Status = Job.StatusEnum.Failed;
                return;
            }

            if (jenkinsJob.lastSuccessfulBuild != null)
            {
                Jenkins.SaveLastConsole(jenkinsJob.name, Path.Combine(job.WorkingDirectory, "consoleText.txt"));
                var success = SaveWorkspace(jenkinsJob, job);

                if (WipeWorkspaceOnSuccess)
                {
                    // wipe workspace before we delete the job
                    Jenkins.WipeWorkSpace(jenkinsJob.name);
                }

                if (DeleteJobOnSuccess)
                {
                    // delete job from server
                    Jenkins.DeleteJob(jenkinsJob.name);
                }

                job.Status = success ? Job.StatusEnum.Succeeded : Job.StatusEnum.FailedToDownload;
            }
            else if (jenkinsJob.lastFailedBuild != null)
            {
                Jenkins.SaveLastConsole(jenkinsJob.name, Path.Combine(job.WorkingDirectory, "consoleText.txt"));
                File.Copy(Path.Combine(job.WorkingDirectory, "consoleText.txt"), Path.Combine(job.WorkingDirectory, LocalPool.Failed));

                try
                {
                    var success = SaveWorkspace(jenkinsJob, job);
                    if (success)
                    {
                        job.Status = Job.StatusEnum.FailedExecution;
                    }
                }
                catch (Exception e)
                {
                    Trace.TraceError("Error retrieving results for {0} ({1}): {2}", job.Title, jenkinsJob.name, e.ToString());
                    job.Status = Job.StatusEnum.FailedToDownload;
                }
            }
            else if (jenkinsJob.lastCompletedBuild != null)
            {
                // build got cancalled
                Jenkins.SaveLastConsole(jenkinsJob.name, Path.Combine(job.WorkingDirectory, "consoleText.txt"));
                File.Copy(Path.Combine(job.WorkingDirectory, "consoleText.txt"), Path.Combine(job.WorkingDirectory, LocalPool.Failed));
                job.Status = Job.StatusEnum.FailedAbortOnServer;
            }
            else if (jenkinsJob.inQueue)
            {
                job.Status = Job.StatusEnum.QueuedOnServer;

                if (jenkinsJob.queueItem != null)
                {
                    if (jenkinsJob.queueItem.stuck)
                    {
                        //Trace.TraceError(Newtonsoft.Json.JsonConvert.SerializeObject(jenkinsJob.queueItem));
                        Trace.TraceError("In queue reason: {0}", jenkinsJob.queueItem.why);
                    }
                    else
                    {
                        Trace.TraceWarning("In queue reason: {0}", jenkinsJob.queueItem.why);
                    }
                }
            }
            else if (jenkinsJob.nextBuildNumber > 1)
            {
                if (jenkinsJob.lastBuild != null &&
                    jenkinsJob.lastBuild.number < jenkinsJob.nextBuildNumber)
                {
                    job.Status = Job.StatusEnum.RunningOnServer;
                }
            }
            // TODO: how to detect if it got cancelled in the queue, before it runs?
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="jenkinsJob"></param>
        /// <param name="job"></param>
        /// <returns>Indicates if job failed or not.</returns>
        private bool SaveWorkspace(
            global::JobManager.Jenkins.Job.Job jenkinsJob,
            Job job)
        {
            job.Status = Job.StatusEnum.DownloadResults;

            bool result = Jenkins.DownloadFileFromVF(Path.Combine(job.WorkingDirectory, "workspace.zip"), jenkinsJob.name, delegate(int percent)
            {
                lvJobQueue.BeginInvoke((MethodInvoker)delegate()
                {
                    ListViewItem item = lvJobQueue.Items.Cast<ListViewItem>().
                       FirstOrDefault(x => x.SubItems[(int)Headers.Id].Text == job.Id.ToString());
                    if (item != null)
                    {
                        item.SubItems[(int)Headers.Progress].Text = percent + "%";
                    }
                });
            });

            if (result == false)
            {
                job.Status = Job.StatusEnum.FailedToDownload;
                return false;
            }

            // unzip package
            string unzipPy = Path.Combine(job.WorkingDirectory, "unzip.py");
            if (File.Exists(unzipPy) == false)
            {
                using (StreamWriter writer = new StreamWriter(unzipPy))
                {
                    writer.WriteLine(@"#!/usr/bin/py
import os
import os.path
import sys
import shutil
import zipfile
try:
    parent_dir_name = os.path.basename(os.getcwd())

    zip = zipfile.ZipFile('workspace.zip')

    # OLD version zip.namelist()[0] is unpredictable
    #root_src_dir = zip.namelist()[0] + parent_dir_name
    # ASSUMPTION workspace.zip has always the parent_dir_name as a zipped directory
    root_src_dir = parent_dir_name

    print root_src_dir
    for entry in zip.infolist():
        if entry.filename.startswith(root_src_dir):
            dest = entry.filename[len(root_src_dir)+1:]
            if dest == '':
                continue
            if dest.endswith('/'):
                if not os.path.isdir(dest):
                    os.mkdir(dest)
            else:
                entry.filename = dest
                zip.extract(entry)
except Exception as msg:
    import traceback
    sys.stderr.write(traceback.format_exc())
    with open('_FAILED.txt', 'wb') as f_out:
        f_out.write(str(msg))
        f_out.write('\nMost likely due to a too long file-path for Windows (max 260).')
    if os.name == 'nt':
        os._exit(3)
    elif os.name == 'posix':
        os._exit(os.EX_OSFILE)

");
                }
            }

            // call unzip.py to unzip the package
            ProcessStartInfo psi = new ProcessStartInfo(Path.Combine(SSHConnection.MetaPathValue, @"bin\Python27\Scripts\python.exe"))
            {
                Arguments = "\"" + unzipPy + "\"",
                WorkingDirectory = job.WorkingDirectory,
                WindowStyle = ProcessWindowStyle.Hidden,
                UseShellExecute = false,
                CreateNoWindow = true,
                RedirectStandardError = true,
            };

            Process proc = new Process()
            {
                StartInfo = psi,
            };

            proc.Start();
            string stderr = proc.StandardError.ReadToEnd();
            proc.WaitForExit();

            if (proc.ExitCode != 0)
            {
                string logFilename = System.IO.Path.Combine(job.WorkingDirectory, LocalPool.Failed);
                File.WriteAllText(logFilename, "unzip.py failed:\n" + stderr);
                return false;
            }
            else
            {
                if (File.Exists(System.IO.Path.Combine(job.WorkingDirectory, LocalPool.Failed)))
                {
                    return false;
                }
                return true;
            }
        }

        protected virtual bool IsFileLocked(FileInfo file)
        {
            FileStream stream = null;

            try
            {
                stream = file.Open(FileMode.Open, FileAccess.ReadWrite, FileShare.None);
            }
            catch (IOException)
            {
                //the file is unavailable because it is:
                //still being written to
                //or being processed by another thread
                //or does not exist (has already been processed)
                return true;
            }
            finally
            {
                if (stream != null)
                    stream.Close();
            }

            //file is not locked
            return false;
        }

        public System.Threading.Timer JenkinsTimer { get; set; }

        Jenkins.Jenkins Jenkins;
        Dictionary<Jenkins.Job.Job, Job> JobMap;

        public string TraceFileName { get; set; }
        private string password { get; set; }
        JobManagerEntities1 entities;

        public JobManager(JobServer server, SSHConnection sshConnection, Dictionary<string, string> settings = null)
        {
            string dbPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), "JobManager.db");
            // entities = new JobManagerEntities1(String.Format("Data Source={0};Version=3;", dbPath));
            SQLiteConnection cnn = new SQLiteConnection(String.Format("data source={0}", dbPath));
            var command = cnn.CreateCommand();
            cnn.Open();
            command.CommandText = @"CREATE TABLE IF NOT EXISTS [SavedJobs] (
                [Title] text NOT NULL,
                [WorkingDirectory] text NOT NULL,
                [RunCommand] text NOT NULL,
                [Status] text NOT NULL,
                [JobName] text PRIMARY KEY NOT NULL
                );";
            command.ExecuteNonQuery();
            command.CommandText = @"PRAGMA user_version;";
            Int64 dbVersion = (Int64) command.ExecuteScalar();
            if (dbVersion == 0)
            {
                command.CommandText = @"ALTER TABLE SavedJobs ADD COLUMN VFUrl TEXT;";
                command.ExecuteNonQuery();
                command.CommandText = @"PRAGMA user_version = 1;";
                command.ExecuteNonQuery();
            }
            cnn.Close();

            entities = new JobManagerEntities1(String.Format("metadata=res://*/SavedJobs.csdl|res://*/SavedJobs.ssdl|res://*/SavedJobs.msl;provider=System.Data.SQLite;provider connection string=\"data source={0}\"", dbPath));
            // Fail fast
            entities.SavedJobs.ToList();

            this.server = server;
            this.sshConnection = sshConnection;

            InitializeComponent();

            this.toolStripStatusLabel = new SpringLabel();
            this.statusStrip1.Items.Add(this.toolStripStatusLabel);

            this.remoteServiceStatusForm = new RemoteServiceStatusForm();

            if (settings != null)
            {
                string value = string.Empty;
                if (settings.TryGetValue("-u", out value))
                {
                    Properties.Settings.Default.UserID = value;
                }
                if (settings.TryGetValue("-p", out value))
                {
                    password = value;
                }
                if (settings.TryGetValue("-U", out value))
                {
                    Properties.Settings.Default.VehicleForgeUri = value;
                }
            }

            //this.TempDir = Path.Combine(System.IO.Path.GetTempPath(), "META_JobManager");

            this.TempDir = Path.Combine(
                System.IO.Path.GetTempPath(),
                Path.Combine("META", "JobManager"));


            if (Directory.Exists(this.TempDir) == false)
            {
                Directory.CreateDirectory(this.TempDir);
            }

            this.TraceFileName = Path.Combine(this.TempDir, "JobManagerLog.trace.txt");

            this.logToolStripMenuItem.ToolTipText += string.Format(" ({0})", Path.GetFullPath(this.TraceFileName));

            this.Jenkins = new Jenkins.Jenkins(this.TempDir);
            this.JobMap = new Dictionary<Jenkins.Job.Job, Job>();

            this.InitJobQueue();

            var fs = new FileStream(this.TraceFileName, FileMode.Create);
            TraceListener fileTL = new TextWriterTraceListener(fs)
            {
                TraceOutputOptions = TraceOptions.DateTime
            };
            // use TraceXXX to get timestamp per http://stackoverflow.com/questions/863394/add-timestamp-to-trace-writeline

            Trace.AutoFlush = true;
            Trace.Listeners.Add(fileTL);

            Trace.TraceInformation(META.Logger.Header());

            this.FormClosing += (sender, args) =>
            {
                SoTTodo.Add(null);
            };
            this.Resize += new EventHandler(JobManager_Resize);
            NotifyIcon = new NotifyIcon();
            NotifyIcon.Icon = Icon;
            NotifyIcon.Visible = false;
            NotifyIcon.Click += new EventHandler(NotifyIcon_Click);

            this.server.JobAdded += JobAdded;
            this.server.SoTAdded += SoTAdded;
            //pool.JobStatusChanged += JobStatusChanged;
            this.timer.Start();

            // load saved settings
            Uri cmd = new Uri(Properties.Settings.Default.CommandUri);
            Uri matlab = new Uri(Properties.Settings.Default.MatLabUri);
            Uri cad = new Uri(Properties.Settings.Default.CADUri);

            Dictionary<Job.TypeEnum, JobManager.TargetMachine> config = new Dictionary<Job.TypeEnum, JobManager.TargetMachine>();
            JobManager.TargetMachine.TargetMachineType type = JobManager.TargetMachine.TargetMachineType.Local;
            if (cmd.Host.Equals("localhost"))
            {
                type = JobManager.TargetMachine.TargetMachineType.Local;
            }
            else
            {
                type = JobManager.TargetMachine.TargetMachineType.Remote;
            }
            config.Add(Job.TypeEnum.Command, new JobManager.TargetMachine(cmd, type));

            if (matlab.Host.Equals("localhost"))
            {
                type = JobManager.TargetMachine.TargetMachineType.Local;
            }
            else
            {
                type = JobManager.TargetMachine.TargetMachineType.Remote;
            }
            config.Add(Job.TypeEnum.Matlab, new JobManager.TargetMachine(matlab, type));

            if (cad.Host.Equals("localhost"))
            {
                type = JobManager.TargetMachine.TargetMachineType.Local;
            }
            else
            {
                type = JobManager.TargetMachine.TargetMachineType.Remote;
            }
            config.Add(Job.TypeEnum.CAD, new JobManager.TargetMachine(cad, type));

            UpdateRuntimeConfig(config);

            this.JobsToBeStartedTimer = new System.Threading.Timer(
                this.JobsToBeStartedCleaner,
                null,
                5000,
                500);

            Shown += new EventHandler(delegate(object o, EventArgs args)
                {
                    if (this.Configure(password) == DialogResult.OK)
                    {
                        lock (this)
                        {
                            this.configured = true;
                            if (server.IsRemote)
                            {
                                this.toolStripStatusLabel.Text = "Configured for remote execution.";
                                this.remoteStatusToolStripMenuItem.Enabled = true;
                                this.remoteStatusToolStripMenuItem.Visible = true;
                            }
                            else
                            {
                                this.toolStripStatusLabel.Text = "Configured for local execution.";
                                this.remoteStatusToolStripMenuItem.Enabled = false;
                                this.remoteStatusToolStripMenuItem.Visible = false;
                            }
                            if (server.IsRemote)
                            {
                                // retreiving saved jobs and adding to the job queue and jobmap
                                foreach (SavedJob saved in entities.SavedJobs)
                                {
                                    if (string.IsNullOrEmpty(saved.VFUrl) == false && server.JenkinsUrl != saved.VFUrl)
                                        continue;
                                    Job job = new Job(server)
                                    {
                                        Title = saved.Title,
                                        Status = Job.StatusEnum.StartedOnServer, // TODO: wrong, but not a big deal, will be re-read from the server
                                        RunCommand = saved.RunCommand,
                                        WorkingDirectory = saved.WorkingDirectory,
                                    };
                                    try
                                    {
                                        Directory.CreateDirectory(saved.WorkingDirectory);
                                    }
                                    catch (Exception e)
                                    {
                                        Trace.TraceError(e.ToString());
                                    }
                                    var jenkinsJobInfo = Jenkins.GetJobInfo(saved.JobName);
                                    if (jenkinsJobInfo == null)
                                    {
                                        Trace.TraceInformation("Saved job {0} ({1}) no longer exists on server. Marking failed.", job.Title, jenkinsJobInfo);
                                        job.Status = Job.StatusEnum.FailedAbortOnServer;
                                    }
                                    else
                                    {
                                        Trace.TraceInformation("Adding saved job {0} ({1})", job.Title, jenkinsJobInfo);
                                        JobMap.Add(jenkinsJobInfo, job);
                                        server.AddJob(job);
                                    }
                                    entities.SavedJobs.Detach(saved);
                                }

                                // start timer for remote execution monitoring
                                this.RestartMonitorTimer(1000);
                            }
                        }
                    }
                    else
                    {
                        this.toolStripStatusLabel.Text = "Closing";
                        this.Close();
                    }
                });

            Trace.TraceInformation("Application started.");
        }


        private void InitJobQueue()
        {
            lvJobQueue.HeaderStyle = ColumnHeaderStyle.Clickable;
            lvJobQueue.View = View.Details;

            lvJobQueue.Columns.Add(Headers.Id.ToString(), Headers.Id.ToString());
            lvJobQueue.Columns[Headers.Id.ToString()].Width = 30;

            lvJobQueue.Columns.Add(Headers.Title.ToString(), Headers.Title.ToString());
            lvJobQueue.Columns[Headers.Title.ToString()].Width = 200;

            lvJobQueue.Columns.Add(Headers.Status.ToString(), Headers.Status.ToString());
            lvJobQueue.Columns[Headers.Status.ToString()].Width = 200;

            lvJobQueue.Columns.Add(Headers.Progress.ToString(), Headers.Progress.ToString());
            lvJobQueue.Columns[Headers.Progress.ToString()].Width = 40;

            lvJobQueue.Columns.Add(Headers.RunCommand.ToString(), Headers.RunCommand.ToString());
            lvJobQueue.Columns[Headers.RunCommand.ToString()].Width = 200;

            lvJobQueue.Columns.Add(Headers.WorkingDirectory.ToString(), Headers.WorkingDirectory.ToString());
            lvJobQueue.Columns[Headers.WorkingDirectory.ToString()].Width = 400;

            //lvJobQueue.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
        }

        private bool configured = false;
        private List<Action> jobsToBeStarted = new List<Action>();
        private System.Threading.Timer JobsToBeStartedTimer { get; set; }


        private void JobsToBeStartedCleaner(object parameter)
        {
            List<Action> jobsToBeStarted = new List<Action>();
            lock (this)
            {
                if (this.configured)
                {
                    jobsToBeStarted = this.jobsToBeStarted;
                    this.jobsToBeStarted = new List<Action>();
                }
            }
            foreach (Action job in jobsToBeStarted)
            {
                try
                {
                    job.Invoke();
                }
                catch (Exception e)
                {
                    Trace.TraceError("Error posting job: " + e.ToString());
                }
            }
        }

        public void JobAdded(Job job)
        {
            Trace.TraceInformation("Job added: {0} {1}", job.Id, job.Title);

            lock (this)
            {
                job.JobStatusChanged += JobStatusChanged;
                if (job.Status == Job.StatusEnum.Ready)
                {
                    jobsToBeStarted.Add(delegate() { StartJob(job); });
                }
                lvJobQueue.Invoke((MethodInvoker)delegate
                {
                    var item = new ListViewItem(new string[] 
									{
										job.Id.ToString(),
										job.Title,
										job.Status.ToString(),
                                        "0%",
										job.RunCommand,
										job.WorkingDirectory
									});
                    for (int i = 0; i < lvJobQueue.Columns.Count; i++)
                    {
                        item.SubItems[i].Name = lvJobQueue.Columns[i].Name;
                    }
                    lvJobQueue.Items.Add(item);
                    AutoResizeColumns();
                });
            }
        }

        Thread SoTWorkerThread = null;
        System.Collections.Concurrent.BlockingCollection<Action> SoTTodo = new System.Collections.Concurrent.BlockingCollection<Action>();

        Semaphore s = new Semaphore(1, 1);
        List<SoT> SoTs = new List<SoT>();
        public void SoTAdded(SoT sot)
        {
            Trace.TraceInformation("SoT added.");
            lock (this)
            {
                if (SoTWorkerThread == null)
                {
                    SoTWorkerThread = new System.Threading.Thread(
                    delegate()
                    {
                        while (true)
                        {
                            s.WaitOne();
                            Action action = SoTTodo.Take();
                            if (action == null)
                            {
                                s.Release();
                                return;
                            }
                            else
                            {
                                action.Invoke();
                                s.Release();
                            }
                        }
                    }
                    );
                    SoTWorkerThread.SetApartmentState(System.Threading.ApartmentState.STA);
                    SoTWorkerThread.Start();
                    SoTWorkerThread.Name = "STA thread for SoT.Run";
                }
                sot.SoTTodo = SoTTodo;
                SoTs.Add(sot);
                jobsToBeStarted.Add(delegate()
                {
                    SoTTodo.Add(delegate { sot.Run(); });
                });
            }
        }

        const int AUTORESIZE_COLUMN_FREQUENCY = 32;
        int autoresize_counter = 10;

        private void AutoResizeColumns()
        {
            if (autoresize_counter-- < 0)
            {
                autoresize_counter = AUTORESIZE_COLUMN_FREQUENCY;
                lvJobQueue.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
                lvJobQueue.Columns[(int)Headers.Progress].Width = Math.Max(lvJobQueue.Columns[(int)Headers.Progress].Width, 40);
            }
        }

        public void StartJob(Job job)
        {
            this.UpdateServiceStatus();
            Run(job);
        }

        private void Run(Job job)
        {
            string failedLog = Path.Combine(job.WorkingDirectory, LocalPool.Failed);
            File.Delete(failedLog);

            TargetMachine target = runtimeConfig[job.Type];

            if (string.IsNullOrEmpty(job.RunCommand) ||
                (job.RunCommand.Split(' ').Count() > 1 &&
                File.Exists(Path.Combine(job.WorkingDirectory, job.RunCommand.Split(' ').FirstOrDefault())) == false))
            {
                Trace.TraceError("Job will not be executed because the runCommand is empty or does not exist: {0}", Path.Combine(job.WorkingDirectory, job.RunCommand));
                using (StreamWriter writer = new StreamWriter(failedLog))
                {
                    writer.WriteLine("ERROR: Job will not be executed because the runCommand is empty or does not exist: {0}", Path.Combine(job.WorkingDirectory, job.RunCommand));
                }
                job.Status = Job.StatusEnum.FailedExecution;
                return;
            }

            if (Properties.Settings.Default.RemoteExecution)
            {
                Trace.TraceInformation("Prepare Job for remote execution: {0} {1}", job.Id, job.Title);

                try
                {
                    // zip working directory
                    string zipFile = Path.Combine(job.WorkingDirectory, "source_data.zip");
                    string zipPy = Path.Combine(job.WorkingDirectory, "zip.py");

                    // if zip.py does not exist create
                    if (File.Exists(zipPy) == false)
                    {
                        using (StreamWriter writer = new StreamWriter(zipPy))
                        {
                            writer.WriteLine(@"#!/usr/bin/python

import zipfile
import os

output_filename = 'source_data.zip'

if os.path.exists(output_filename):
    os.remove(output_filename)

with zipfile.ZipFile(output_filename, 'w') as z:
    parent_dir_name = os.path.basename(os.getcwd())
    os.chdir('..\\')
    for dirpath,dirs,files in os.walk(parent_dir_name):
      for f in files:
        if output_filename == f:
            continue
        fn = os.path.join(dirpath, f)
        #print fn
        z.write(fn, compress_type=zipfile.ZIP_DEFLATED)");
                        }
                    }

                    if (File.Exists(zipFile) == false)
                    {
                        // call zip.py to zip the package if it does not exist
                        ProcessStartInfo psi = new ProcessStartInfo(Path.Combine(SSHConnection.MetaPathValue, @"bin\Python27\Scripts\python.exe"))
                        {
                            Arguments = "\"" + zipPy + "\"",
                            WorkingDirectory = job.WorkingDirectory,
                            WindowStyle = ProcessWindowStyle.Hidden,
                            UseShellExecute = false,
                            CreateNoWindow = true,
                            RedirectStandardError = true,
                        };

                        Process proc = new Process()
                        {
                            StartInfo = psi,
                        };

                        proc.Start();
                        string stderr = proc.StandardError.ReadToEnd();
                        proc.WaitForExit();
                        if (proc.ExitCode != 0)
                        {
                            job.Status = Job.StatusEnum.Failed;
                            Trace.TraceError("zip.py failed with exit code {0}. stderr was {1}", proc.ExitCode, stderr);
                            return;
                        }
                    }

                    if (File.Exists(zipFile) == false)
                    {
                        job.Status = Job.StatusEnum.Failed;
                        Trace.TraceError("zip.py did not produce {0}", zipFile);
                        return;
                    }

                    if (this.IsServiceOnline() == false)
                    {
                        // put all job into a waiting status if the server is disconnected
                        job.Status = Job.StatusEnum.WaitingForStart;

                        // put the job back to the list and try to start it again later
                        jobsToBeStarted.Add(() => { job.Start(); });
                        return;
                    }

                    // create a job on the remote machine

                    // KMS: add random hex number because users can't see each other's Jenkins jobs
                    var randomid = global::JobManager.Jenkins.Jenkins.GetRandomHexNumber(8);
                    string jobname = string.Format("{0}_{1}", Properties.Settings.Default.UserID, randomid);
                    string description = string.Format("{0}_{1}_{2:00000}_{3}", Properties.Settings.Default.UserID, job.Title,
                        job.Id, randomid);
                    jobname = jobname.Replace(' ', '_');

                    //if (Jenkins.JobExists(jobname))
                    {
                        // job already exists
                        //   Jenkins.DeleteJob(jobname);
                    }


                    string cmd = Path.GetExtension(job.RunCommand) == ".py" ?
                        "python " + job.RunCommand :
                        job.RunCommand;

                    string labels = job.Labels;

                    job.Status = Job.StatusEnum.UploadPackage;

                    string zipFileUrl = Jenkins.UploadFileToVF(zipFile, jobname, delegate(int percent)
                    {
                        lvJobQueue.BeginInvoke((MethodInvoker)delegate()
                        {
                            ListViewItem item = lvJobQueue.Items.Cast<ListViewItem>().
                               FirstOrDefault(x => x.SubItems[(int)Headers.Id].Text == job.Id.ToString());
                            if (item != null)
                            {
                                item.SubItems[(int)Headers.Progress].Text = percent + "%";
                            }
                        });
                    });

                    if (zipFileUrl == null)
                    {
                        job.Status = Job.StatusEnum.FailedToUploadServer;
                        return;
                    }

                    // 1. {0} description
                    // 2. {1} working directory name
                    // 3. {2} runcommand that needs to be executed in the working directory
                    // 4. {3} required slave node (Label expression)
                    // 5. {4} REMOVED: pattern to archiving artifacts.
                    // 6. {5} zip.py server side hook
                    // 7. {6} name of the zipped results file
                    string resultZipName = "results.zip";
                    string resultZipPy = string.Format(job.ResultsZip, resultZipName, Path.GetFileName(job.WorkingDirectory));
                    var config_xml = String.Format(
                            Properties.Resources.job_config,
                            SecurityElement.Escape(description),
                            Path.GetFileName(job.WorkingDirectory),
                            SecurityElement.Escape(cmd),
                            SecurityElement.Escape(labels),
                            "",
                            SecurityElement.Escape(resultZipPy),
                            SecurityElement.Escape(resultZipName));

                    var jobinfonew = Jenkins.CreateJob(jobname, config_xml);

                    if (jobinfonew == null)
                    {
                        // job creation failed
                        job.Status = Job.StatusEnum.FailedToUploadServer;
                        return;
                    }

                    job.Status = Job.StatusEnum.PostedToServer;

                    string returned_config_xml = Jenkins.GetJobConfig(jobname);
                    if (returned_config_xml != null)
                    {
                        if (returned_config_xml.IndexOf("org.jvnet.hudson.plugins.Jython") == -1)
                        {
                            string logFilename = System.IO.Path.Combine(job.WorkingDirectory, LocalPool.Failed);
                            File.WriteAllText(logFilename, "Jenkins does not have the Jython plugin installed");
                            Trace.TraceError("Jenkins does not have the Jython plugin installed");
                            job.Status = Job.StatusEnum.FailedToUploadServer;
                            return;
                        }
                    }
                    else ; // FIXME throw?

                    // send zip and start job
                    if (Jenkins.BuildJob(jobname, jobname, job.BuildQuery) == null)
                    {
                        job.Status = Job.StatusEnum.FailedToUploadServer;
                        return;
                    }

                    JobMap.Add(Jenkins.GetJobInfo(jobname), job);
                    job.Status = Job.StatusEnum.StartedOnServer;
                }
                catch (Exception ex)
                {
                    Jenkins.DebugLog.WriteLine(ex.ToString().Replace("\n", "  "));
                    job.Status = Job.StatusEnum.FailedToUploadServer;
                    MessageBox.Show(ex.ToString(), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    Trace.TraceError(ex.ToString());
                }

            }
            else
            {
                // if local
                pool.EnqueueJob(job);
            }
        }


        public void JobStatusChanged(Job job)
        {
            try
            {
                Trace.TraceError("Job {0} changed status to {1}", job.Title, job.Status.ToString());
                autoresize_counter--;
                lvJobQueue.Invoke((MethodInvoker)delegate
                {
                    Color color = Color.White;
                    switch (job.Status)
                    {
                        case Job.StatusEnum.Ready:
                            color = Color.PeachPuff;
                            break;
                        case Job.StatusEnum.QueuedLocal:
                        case Job.StatusEnum.QueuedOnServer:
                        case Job.StatusEnum.StartedOnServer:
                            color = Color.LightYellow;
                            break;
                        case Job.StatusEnum.UploadPackage:
                        case Job.StatusEnum.DownloadResults:
                            color = Color.PaleTurquoise;
                            break;
                        case Job.StatusEnum.PostedToServer:
                            color = Color.Khaki;
                            break;
                        case Job.StatusEnum.RunningOnServer:
                        case Job.StatusEnum.RunningLocal:
                            color = Color.LightSkyBlue;
                            break;
                        case Job.StatusEnum.Succeeded:
                            color = Color.LightGreen;
                            break;
                        case Job.StatusEnum.Failed:
                        case Job.StatusEnum.FailedAbortOnServer:
                        case Job.StatusEnum.FailedExecution:
                        case Job.StatusEnum.FailedToUploadServer:
                        case Job.StatusEnum.FailedToDownload:
                            color = Color.Tomato;
                            break;
                        default:
                            color = Color.White;
                            break;
                    };

                    if (server.IsRemote)
                    {
                        if (job.Status == Job.StatusEnum.StartedOnServer)
                        {
                            Jenkins.Job.Job jenkinsJob = JobMap.Where(kvp => kvp.Value == job).FirstOrDefault().Key;
                            if (jenkinsJob == null)
                            {
                                Trace.TraceError("Could not find jenkins job for {0}. Persistence will not work", job.Title);
                            }
                            else
                            {
                                SavedJob saved = SavedJob.CreateSavedJob(
                                    job.Title, job.WorkingDirectory, job.RunCommand, job.Status.ToString(), jenkinsJob.name);
                                saved.VFUrl = server.JenkinsUrl;
                                entities.SavedJobs.AddObject(saved);
                                entities.SaveChanges();
                                entities.SavedJobs.Detach(saved);
                            }
                        }
                        else if (job.IsFailed() || job.Status == Job.StatusEnum.Succeeded)
                        {
                            Jenkins.Job.Job jenkinsJob = JobMap.Where(kvp => kvp.Value == job).FirstOrDefault().Key;
                            if (jenkinsJob == null)
                            {
                                Trace.TraceError("Could not find jenkins job for {0}. It may remain in the database", job.Title);
                            }
                            else
                            {
                                foreach (var entity in entities.SavedJobs.Where(x => x.JobName == jenkinsJob.name))
                                {
                                    entities.SavedJobs.DeleteObject(entity);
                                }
                                entities.SaveChanges();
                            }
                        }
                    }

                    var item = lvJobQueue.Items.
                        Cast<ListViewItem>().
                        FirstOrDefault(x => x.SubItems[(int)Headers.Id].Text == job.Id.ToString());

                    item.BackColor = color;
                    item.SubItems[(int)Headers.Status].Text = job.Status.ToString();

                    AutoResizeColumns();
                });

                if (job.Status == Job.StatusEnum.Ready)
                {
                    jobsToBeStarted.Add(delegate() { StartJob(job); });
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                Trace.TraceError(ex.ToString());
            }

        }

        void NotifyIcon_Click(object sender, EventArgs e)
        {
            if ((e as MouseEventArgs).Button == System.Windows.Forms.MouseButtons.Left)
            {
                WindowState = FormWindowState.Normal;
                Show();
                NotifyIcon.Visible = false;
            }
        }

        void JobManager_Resize(object sender, EventArgs e)
        {
            if (WindowState == FormWindowState.Minimized)
            {
                //NotifyIcon.Visible = true;
                //Hide();
            }
        }

        class ListViewItemComparer : System.Collections.IComparer
        {
            private int col;
            public ListViewItemComparer(int column)
            {
                col = column;
            }
            public int Compare(object x, object y)
            {
                int returnVal = -1;
                returnVal = String.Compare(((ListViewItem)x).SubItems[col].Text,
                ((ListViewItem)y).SubItems[col].Text);
                return returnVal;
            }
        }

        private void lvJobQueue_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            lvJobQueue.ListViewItemSorter = new ListViewItemComparer(e.Column);
            lvJobQueue.Sort();
        }

        void lvJobQueue_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (e.KeyData == (Keys.C | Keys.Control))
            {
                ListViewItem item = (sender as ListView).FocusedItem;
                if (item != null)
                {
                    Clipboard.SetText(item.SubItems[4].Text);
                }
            }
        }

        private void lvJobQueue_SelectedIndexChanged(object sender, EventArgs e)
        {
            var lv = sender as ListView;
            ListViewItem item = lv.FocusedItem;
            if (item != null)
            {
                string status = item.SubItems[Headers.Status.ToString()].Text;
                showInExplorerToolStripMenuItem.Enabled = true;
            }

            var selectedJobs = lv.SelectedItems.Cast<ListViewItem>().Select(x => server.Jobs.FirstOrDefault(job => job.Id.ToString() == x.SubItems[Headers.Id.ToString()].Text));

            openLogFileToolStripMenuItem.Enabled = selectedJobs.All(x =>
                x.IsFailed() &&
                File.Exists(Path.Combine(x.WorkingDirectory, LocalPool.Failed)));

            rerunToolStripMenuItem.Enabled = selectedJobs.All(x =>
                (x.IsFailed() || x.Status == Job.StatusEnum.Succeeded) &&
                x.RerunEnabled &&
                Directory.Exists(x.WorkingDirectory));
        }

        private bool WipeWorkspaceOnSuccess { get { return global::JobManager.Properties.Settings.Default.WipeWorkspaceOnSuccess; } }
        private bool DeleteJobOnSuccess { get { return global::JobManager.Properties.Settings.Default.DeleteJobOnSuccess; } }
        public SSHConnection sshConnection;


        private DialogResult Configure(string password = null)
        {
            using (Configuration cfgRemote = new Configuration(sshConnection, Jenkins, password))
            {
                this.toolStripStatusLabel.Text = "Configuring";
                if (string.IsNullOrEmpty(password))
                {
                    var dr = cfgRemote.ShowDialog(this);
                    server.JenkinsUrl = Properties.Settings.Default.VehicleForgeUri;
                    server.UserName = Properties.Settings.Default.UserID;
                    server.IsRemote = Properties.Settings.Default.RemoteExecution;
                    return dr;
                }
                else
                {
                    cfgRemote.btnSave_Click(cfgRemote.btnSave, null);
                    server.JenkinsUrl = Properties.Settings.Default.VehicleForgeUri;
                    server.UserName = Properties.Settings.Default.UserID;
                    server.IsRemote = Properties.Settings.Default.RemoteExecution;
                    return cfgRemote.DialogResult;
                }
            }
        }

        private void logToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (File.Exists(this.TraceFileName))
            {
                Process.Start(this.TraceFileName);
            }
        }

        private void createJobToolStripMenuItem_Click(object sender, EventArgs e)
        {
            CreateJob cj = new CreateJob();
            cj.Show(this);
        }

        private void showInExplorerToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ListViewItem item = lvJobQueue.FocusedItem;
            if (item != null)
            {
                string directory = item.SubItems[Headers.WorkingDirectory.ToString()].Text;
                if (System.IO.Directory.Exists(directory))
                {
                    System.Diagnostics.Process.Start(directory);
                }
                else
                {
                    MessageBox.Show(String.Format("Directory does not exist. {0}", directory));
                }
                lvJobQueue.FocusedItem = item;
            }
        }

        private void openLogFileToolStripMenuItem_Click(object sender, EventArgs e)
        {
            var toOpenLog = lvJobQueue
                .SelectedItems
                .Cast<ListViewItem>()
                .Select(x =>
                    server.Jobs.FirstOrDefault(job => job.Id.ToString() == x.SubItems[Headers.Id.ToString()].Text))
                .Where(x => x.IsFailed());

            foreach (var item in toOpenLog)
            {
                string directory = item.WorkingDirectory;
                string logFile = System.IO.Path.Combine(directory, LocalPool.Failed);
                if (System.IO.File.Exists(logFile))
                {
                    System.Diagnostics.Process.Start(logFile);
                }
                else
                {
                    MessageBox.Show(String.Format("Log file does not exist. {0}", logFile));
                }
            }
        }

        private void rerunToolStripMenuItem_Click(object sender, EventArgs e)
        {
            var toReRun = lvJobQueue
                .SelectedItems
                .Cast<ListViewItem>()
                .SelectMany(x =>
                    server.Jobs.Where(job => job.Id.ToString() == x.SubItems[Headers.Id.ToString()].Text))
                .Where(x =>
                    (x.IsFailed() || x.Status == Job.StatusEnum.Succeeded) &&
                    x.RerunEnabled);

            foreach (var selectedJob in toReRun)
            {
                try
                {
                    // run it again
                    selectedJob.Status = Job.StatusEnum.WaitingForStart;
                    selectedJob.Start();
                    Trace.TraceInformation("Rerunning job: {0} {1}", selectedJob.Id, selectedJob.Title);
                }
                catch (Exception ex)
                {
                    Trace.TraceError(ex.ToString());
                }
            }

        }

        private void JobManager_Load(object sender, EventArgs e)
        {
            this.toolStripStatusLabel.Text = "Loading";
            ListViewHelper.EnableDoubleBuffer(this.lvJobQueue);
            ListViewHelper.EnableDoubleBuffer(this.remoteServiceStatusForm.lvRemoteNodes);
        }

        private void remoteStatusToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (this.remoteServiceStatusForm == null ||
                this.remoteServiceStatusForm.IsDisposed)
            {
                this.remoteServiceStatusForm = new RemoteServiceStatusForm();
                this.remoteServiceStatusForm.Show();
                this.UpdateServiceStatus();
            }
            else
            {
                this.remoteServiceStatusForm.Hide();
                this.remoteServiceStatusForm.Show();
                this.UpdateServiceStatus();
            }
        }
    }

    /// <summary>
    /// http://stackoverflow.com/questions/2903172/indicate-truncation-in-tooltipstatuslabel-automatically
    /// </summary>
    [ToolStripItemDesignerAvailability(ToolStripItemDesignerAvailability.StatusStrip)]
    public class SpringLabel : ToolStripStatusLabel
    {
        public SpringLabel()
        {
            this.Spring = true;
        }
        protected override void OnPaint(PaintEventArgs e)
        {
            var flags = TextFormatFlags.Left | TextFormatFlags.EndEllipsis;
            var bounds = new Rectangle(0, 0, this.Bounds.Width, this.Bounds.Height);
            TextRenderer.DrawText(e.Graphics, this.Text, this.Font, bounds, this.ForeColor, flags);
        }
    }


    public class DisableKeepAliveWebClient : WebClient
    {
        public bool? AllowReadStreamBuffering { get; set; }

        public DisableKeepAliveWebClient()
        {
        }

        protected override WebRequest GetWebRequest(Uri address)
        {
            WebRequest request = base.GetWebRequest(address);
            HttpWebRequest webRequest = request as HttpWebRequest;
            if (webRequest != null)
            {
                webRequest.Timeout = Jenkins.Jenkins.HTTP_WEB_REQUEST_TIMEOUT;
                webRequest.KeepAlive = false;
                if (AllowReadStreamBuffering != null)
                {
                    this.AllowReadStreamBuffering = AllowReadStreamBuffering;
                }
            }
            return request;
        }
    }

    public static class ExtensionMethods
    {
        /// <summary>
        /// http://stackoverflow.com/questions/808867/invoke-or-begininvoke-cannot-be-called-on-a-control-until-the-window-handle-has
        /// </summary>
        /// <param name="uiElement"></param>
        /// <param name="updater"></param>
        /// <param name="forceSynchronous"></param>
        public static void SafeInvoke(
            this Control uiElement,
            Action updater,
            bool forceSynchronous = false)
        {
            if (uiElement == null)
            {
                throw new ArgumentNullException("uiElement");
            }

            if (uiElement.InvokeRequired)
            {
                if (forceSynchronous)
                {
                    uiElement.Invoke((Action)delegate { SafeInvoke(uiElement, updater, forceSynchronous); });
                }
                else
                {
                    uiElement.BeginInvoke((Action)delegate { SafeInvoke(uiElement, updater, forceSynchronous); });
                }
            }
            else
            {
                if (uiElement.IsDisposed)
                {
                    // Zsolt: return is ok for us
                    return;
                    throw new ObjectDisposedException("Control is already disposed.");
                }

                updater();
            }
        }
    }
}
