/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.IO;
using System.Diagnostics;

namespace JobManager
{
    internal class LocalPool
    {
        /// <summary>
        /// Each job must generate this file if the execution was failed.
        /// </summary>
        public const string Failed = "_FAILED.txt";

        public int NumAllThread { get; private set; }
        public int NumCommandThread { get; private set; }
        public int NumMatLabThread { get; private set; }
        public int NumCADThread { get; private set; }

        /// <summary>
        /// Task - Job pairs.
        /// </summary>
        Dictionary<Task, Job> tasks =
            new Dictionary<Task, Job>();

        /// <summary>
        /// Task factory.
        /// </summary>
        TaskFactory tf { get; set; }

        /// <summary>
        /// Global semaphore limits the maximum number of processes.
        /// </summary>
        Semaphore SemAll { get; set; }

        /// <summary>
        /// Semaphores for different type of jobs.
        /// </summary>
        Dictionary<Job.TypeEnum, Semaphore> SemJob { get; set; }

        /// <summary>
        /// Hold a reference to the currently running/finished user processes.
        /// </summary>
        Dictionary<Job, System.Diagnostics.Process> processes =
            new Dictionary<Job, System.Diagnostics.Process>();

        CancellationTokenSource ts { get; set; }
        CancellationToken ct { get; set; }


        public LocalPool(
            int numCommandThread = 8,
            int numMatLabThread = 4,
            int numCADThread = 2)
        {
            ts = new CancellationTokenSource();
            ct = ts.Token;

            tf = new TaskFactory(
                ct,
                TaskCreationOptions.LongRunning,
                TaskContinuationOptions.None,
                null);

            int numAllThread = numCommandThread + numMatLabThread + numCADThread;

            // do not use more threads than cores
            numAllThread = numAllThread < Environment.ProcessorCount ?
                numAllThread :
                Environment.ProcessorCount;

            NumAllThread = numAllThread;
            NumCommandThread = numCommandThread;
            NumMatLabThread = numMatLabThread;
            NumCADThread = numCADThread;

            SemAll = new Semaphore(numAllThread, numAllThread);
            SemJob = new Dictionary<Job.TypeEnum, Semaphore>();
            SemJob.Add(Job.TypeEnum.Command, new Semaphore(numCommandThread, numCommandThread));
            SemJob.Add(Job.TypeEnum.Matlab, new Semaphore(numMatLabThread, numMatLabThread));
            SemJob.Add(Job.TypeEnum.CAD, new Semaphore(numCADThread, numCADThread));
        }

        ~LocalPool()
        {
            Dispose();
        }

        /// <summary>
        /// Kills all currently running/pending tasks.
        /// Releases all resources.
        /// </summary>
        public void Dispose()
        {
            // kill user processes
            foreach (var item in processes.Values)
            {
                try
                {
                    if (item.HasExited == false)
                    {
                        item.Kill();
                        item.Close();
                    }
                }
                catch (InvalidOperationException ex)
                {
                    System.Diagnostics.Debug.WriteLine(ex);
                }
            }
            // kill pending jobs/tasks
            //ts.Cancel();
            //foreach (var item in tasks)
            //{
            //  item.Key.Wait();
            //}
        }

        public void EnqueueJob(Job j)
        {
            Trace.TraceInformation(string.Format("JobEnqueued in local pool: {0} {1}", j.Id, j.Title));

            Task t = tf.StartNew(() => RunJob(j), ct).
                ContinueWith(job => Done(job));
            

            tasks.Add(t, j);
            
            //Task t = new Task<Job>(() => RunJob(j)).ContinueWith(job => Done(job));

        }

        private Job RunJob(Job job)
        {
            try
            {
                job.Status = Job.StatusEnum.QueuedLocal;
                SemJob[job.Type].WaitOne();
                SemAll.WaitOne();

                // TODO: replace this function body into real 'calls'

                System.Diagnostics.Process proc0 = new System.Diagnostics.Process();
                if (processes.ContainsKey(job))
                {
                    processes.Remove(job);
                }

                processes.Add(job, proc0);

                System.Diagnostics.ProcessStartInfo psi =
                    new System.Diagnostics.ProcessStartInfo();

                psi.Arguments = "/S /C \"" + job.RunCommand + "\"";
                psi.CreateNoWindow = true;
                psi.FileName = "cmd";
                psi.UseShellExecute = false; // true
                psi.WindowStyle = System.Diagnostics.ProcessWindowStyle.Minimized;
                psi.WorkingDirectory = Path.Combine(job.WorkingDirectory);

                psi.RedirectStandardOutput = true;
                psi.RedirectStandardError = true;
                proc0.StartInfo = psi;

                using (StreamWriter stderr = new StreamWriter(Path.Combine(job.WorkingDirectory, "stderr.txt")))
                using (StreamWriter stdout = new StreamWriter(Path.Combine(job.WorkingDirectory, "stdout.txt")))
                    try
                    {
                        proc0.OutputDataReceived += ((sender, e) =>
                        {
                            stdout.Write(e.Data);
                        });
                        proc0.ErrorDataReceived += ((sender, e) =>
                        {
                            stderr.Write(e.Data);
                        });
                        string failedLog = Path.Combine(job.WorkingDirectory, Failed);
                        File.Delete(failedLog);
                        proc0.Start();
                        job.Status = Job.StatusEnum.RunningLocal;
                        if (psi.RedirectStandardOutput)
                        {
                            // begin read only if it was redirected
                            proc0.BeginOutputReadLine();
                        }
                        if (psi.RedirectStandardError)
                        {
                            // begin read only if it was redirected
                            proc0.BeginErrorReadLine();
                        }
                        //JobStatusChanged.Invoke(job);
                        proc0.WaitForExit();

                        if (File.Exists(failedLog))
                        {
                            job.Status = Job.StatusEnum.Failed;
                            //JobStatusChanged.Invoke(job);
                        }
                        else if (proc0.ExitCode == 1)
                        {
                            // command not found
                            using (StreamWriter writer = new StreamWriter(failedLog))
                            {
                                writer.WriteLine("Specified command was not found: {0}",
                                    Path.Combine(job.WorkingDirectory, job.RunCommand));
                            }
                            job.Status = Job.StatusEnum.FailedExecution;
                            //JobStatusChanged.Invoke(job);
                        }
                        else if (proc0.ExitCode != 0)
                        {
                            using (StreamWriter writer = new StreamWriter(failedLog))
                            {
                                writer.WriteLine("Exit Code: " + proc0.ExitCode);
                            }
                            job.Status = Job.StatusEnum.FailedExecution;
                        }
                        proc0.Close();
                    }
                    catch (Exception ex)
                    {
                        job.Status = Job.StatusEnum.Failed;
                        //JobStatusChanged.Invoke(job);
                        System.Diagnostics.Debug.WriteLine(ex);
                        Trace.TraceError(ex.ToString());
                    }
            }
            catch (Exception ex)
            {
                Trace.TraceError(ex.ToString());
            }
            finally
            {
                SemAll.Release();
                SemJob[job.Type].Release();
            }
            return job;
        }



        private void Done(Task<Job> job)
        {
            if (job.Status == TaskStatus.Faulted)
            {
                tasks[job].Status = Job.StatusEnum.FailedExecution;
                //JobStatusChanged.Invoke(job.Result);
            }
            else
            {
                if (job.Result.Status == Job.StatusEnum.RunningLocal)
                {
                    if (job.IsCompleted)
                    {
                        job.Result.Status = Job.StatusEnum.Succeeded;
                    }
                    else if (job.IsFaulted ||
                        job.IsCanceled)
                    {
                        job.Result.Status = Job.StatusEnum.FailedExecution;
                    }
                    //JobStatusChanged.Invoke(job.Result);
                }
            }
        }

    }
}
