/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Linq;

namespace JobManager
{
    public class Job : MarshalByRefObject
    {
        /// <summary>
        /// Global number of jobs.
        /// </summary>
        public static int NumJobs = 0;

        // TODO: change this id to guid
        public int Id { get; private set; }
        public string Title { get; set; }
        public string WorkingDirectory { get; set; }
        public string RunCommand { get; set; }
        public List<Job> Dependencies { get; set; }

        public DateTime TimePosted { get; set; }
        public DateTime TimeStart { get; set; }
        public DateTime TimeDone { get; set; }

        public TimeSpan TimeInQueue { get; set; }
        public TimeSpan TimeTotal { get; set; }

        public bool RerunEnabled { get; set; }

        public Statistics Stat { get; set; }

        private JobServer Server { get; set; }

        public Job(JobServer server)
        {
            this.Server = server;
            this.Id = Job.NumJobs++;
            this.TimePosted = DateTime.Now;
            this.TimeInQueue = TimeSpan.Zero;
            this.TimeTotal = TimeSpan.Zero;

            this.Labels = DefaultLabels;
            this.BuildQuery = DefaultBuildQuery;
            this.ResultsZip = Properties.Resources.zip;

            this.RerunEnabled = true;

            this.Stat = new Statistics()
            {
                Id = this.Id,
                JobReceived = DateTime.Now.ToString("u"),
            };
        }

        /// <summary>
        /// This funciton puts changes the job's status to queued if it was waiting for start.
        /// </summary>
        public void Start()
        {
            if (this.Status == Job.StatusEnum.WaitingForStart)
            {
                this.Status = StatusEnum.Ready;
            }
        }

        public enum StatusEnum
        {
            WaitingForStart,
            Ready,
            QueuedLocal,
            RunningLocal,
            UploadPackage,
            PostedToServer,
            StartedOnServer,
            QueuedOnServer,
            RunningOnServer,
            DownloadResults,
            Succeeded,
            FailedToUploadServer,
            FailedToDownload,
            FailedAbortOnServer,
            FailedExecution,
            Failed,
        }

        public event JobActionHandler JobStatusChanged;

        private StatusEnum status = StatusEnum.Ready;
        public StatusEnum Status
        {
            get { return status; }
            set
            {
                status = value;
                Trace.TraceInformation("Name of Job : {0}, New Status : {1}", this.Title, status);
                
                if (JobStatusChanged != null)
                {
                    JobStatusChanged.Invoke(this);
                }

                // If something goes wrong here, catch any exception and print as trace warning
                try
                {
                    this.UpdateStatistics(status);
                    if (status == StatusEnum.Succeeded || this.IsFailed() || status == StatusEnum.Ready)
                    {
                        this.SaveStatistics();
                    }
                }
                catch (Exception exc)
                {
                    Trace.TraceWarning(exc.ToString());
                }
            }
        }

        private void UpdateStatistics(StatusEnum newStatus)
        {
            Statistics.Execution thisExecution;

            if (newStatus == Job.StatusEnum.Ready)
            {
                thisExecution = new Statistics.Execution();
                this.Stat.Executions.Add(thisExecution);
                thisExecution.StartTime = DateTime.Now.ToString("u");
            }
            else
            {
                thisExecution = this.Stat.Executions.LastOrDefault();
            }

            if (newStatus == StatusEnum.RunningLocal ||
                newStatus == StatusEnum.RunningOnServer)
            {
                TimeStart = DateTime.Now;
                TimeInQueue = TimeStart - TimePosted;
                thisExecution.StartTime = DateTime.Now.ToString("u");
            }
            else if (newStatus == StatusEnum.Succeeded || this.IsFailed())
            {
                TimeDone = DateTime.Now;
                if (TimeDone > TimeStart)
                    TimeTotal = TimeDone - TimeStart;
            }

            // update the execution statistics (queued -> new execution)

            if (thisExecution != null)
            {
                thisExecution.StatusTraces.Add(new Statistics.Execution.StatusTrace()
                {
                    Status = newStatus.ToString(),
                    TimeStamp = DateTime.Now.ToString("u")
                });
            }

        }

        public string SaveStatistics(bool saveToFile = true, string fileName = null)
        {

            this.Stat.JobName = this.Title;
            this.Stat.RunCommand = this.RunCommand;
            this.Stat.UserId = this.Server.UserName;
            this.Stat.IsRemote = this.Server.IsRemote;
            this.Stat.VFUrl = this.Server.JenkinsUrl;

            // This is only used if saving/reading to/from file
            if (string.IsNullOrEmpty(fileName))
            {
                fileName = Path.Combine(this.WorkingDirectory, "stat.json");
            }

            
            // Load stat-file to collect ToolSpecifics
            if (File.Exists(fileName))
            {
                    using (StreamReader reader = new StreamReader(fileName))
                    {
                        var readStatistics = Newtonsoft.Json.JsonConvert.DeserializeObject<Statistics>(
                            reader.ReadToEnd(),
                            new Newtonsoft.Json.Converters.IsoDateTimeConverter()
                            {
                                DateTimeStyles = System.Globalization.DateTimeStyles.AdjustToUniversal
                            });
                        if (readStatistics.ToolSpecifics != null)
                        {
                            //Trace.TraceInformation(readStatistics.ToolSpecifics.ToString());
                            this.Stat.ToolSpecifics = readStatistics.ToolSpecifics;
                            Trace.TraceInformation("Statistics.ToolSpecifics read in from {0}", fileName);
                        }
                        else
                        {
                            Trace.TraceInformation("Statistics.ToolSpecifics = null in stat-file.");
                        }
                    }
            }

            var jsonContent = Newtonsoft.Json.JsonConvert.SerializeObject(
                this.Stat,
                Newtonsoft.Json.Formatting.Indented,
                new Newtonsoft.Json.Converters.IsoDateTimeConverter()
                {
                    DateTimeStyles = System.Globalization.DateTimeStyles.AdjustToUniversal
                });

            if (saveToFile)
            {
                using (StreamWriter writer = new StreamWriter(fileName))
                {
                    writer.WriteLine(jsonContent);
                    Trace.TraceInformation("Statistics saved to {0}", fileName);
                }
            }

            return jsonContent;
        }

        public enum TypeEnum
        {
            Command,
            Matlab,
            CAD,
        }

        public bool IsFailed()
        {
            return status == StatusEnum.FailedExecution ||
                   status == StatusEnum.FailedAbortOnServer ||
                   status == StatusEnum.FailedToUploadServer ||
                   status == StatusEnum.FailedToDownload ||
                   status == StatusEnum.Failed;
        }

        public TypeEnum Type { get; set; }

        public string Labels { get; set; }

        public string BuildQuery { get; set; }

        /// <summary>
        /// zip.py server side hook
        /// </summary>
        public string ResultsZip { get; set; }

        public const string DefaultLabels = "Windows13.09";
        public const string DefaultBuildQuery = "";
    }

    public delegate void JobActionHandler(Job job);
    public delegate void SoTActionHandler(SoT sot);

    public class JobServer : MarshalByRefObject
    {
        public string JenkinsUrl { get; set; }
        public string UserName { get; set; }
        public bool IsRemote { get; set; }

        public Job CreateJob()
        {
            Job job = new Job(this);
            return job;
        }

        public SoT CreateSoT()
        {
            SoT sot = new SoT(this);
            return sot;
        }

        public List<Job> Jobs { get; set; }
        public List<SoT> SoTs { get; set; }

        public void AddJob(Job job)
        {
            if (Jobs == null)
            {
                Jobs = new List<Job>();
            }
            Jobs.Add(job);

            JobAdded.Invoke(job);
        }

        public void AddSoT(SoT sot)
        {
            if (SoTs == null)
            {
                SoTs = new List<SoT>();
            }
            SoTs.Add(sot);

            SoTAdded.Invoke(sot);
        }

        public event JobActionHandler JobAdded;
        public event SoTActionHandler SoTAdded;
        //public event JobActionHandler JobStarted;
        //public event JobActionHandler JobCompleted;

        public override object InitializeLifetimeService()
        {
            return null;
        }
    }

}
