/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿// -----------------------------------------------------------------------
// <copyright file="SoT.cs" company="">
// TODO: Update copyright text.
// </copyright>
// -----------------------------------------------------------------------

namespace JobManager
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.IO;
    using System.Diagnostics;
    using System.Runtime.InteropServices;
    using GME.MGA;
    using System.Threading;

    /// <summary>
    /// TODO: Update summary.
    /// </summary>
    public class SoT : MarshalByRefObject
    {
        public JobServer Server { get; set; }
        public string WorkingDirectory { get; set; }
        public string ProjectConnStr { get; set; }

        public List<Job> Jobs { get; set; }

        public SoT(JobServer server)
        {
            this.Server = server;
        }

        string currentObjectName { get; set; }
        string sotName { get; set; }
        CyPhySoT.SotConfig sotConfig { get; set; }
        Action<Job> JobAction { get; set; }

        MgaProject Project { get; set; }
        CyPhySoT.CyPhySoTInterpreter sotInterpreter { get; set; }
        MgaFCO CurrentObj { get; set; }

        public void Run()
        {
            if (string.IsNullOrEmpty(this.WorkingDirectory) ||
                Directory.Exists(this.WorkingDirectory) == false)
            {
                // nothing needs to be done.
                return;
            }

            // TODO: exception handling... 

            // get manifest file
            string manifestFile = Path.Combine(this.WorkingDirectory, "manifest.sot.json");
            if (File.Exists(manifestFile) == false)
            {
                Trace.TraceError("{0} file does not exist", manifestFile);
                return;
            }

            // this could fail... exception handling!
            sotConfig = Newtonsoft.Json.JsonConvert.DeserializeObject<CyPhySoT.SotConfig>(File.ReadAllText(manifestFile));

            // get mga filename
            string mgaFileName = Path.Combine(this.WorkingDirectory, sotConfig.ProjectFileName);

            if (File.Exists(mgaFileName) == false)
            {
                Trace.TraceError("{0} file does not exist", mgaFileName);
                return;
            }

            this.ProjectConnStr = "MGA=" + mgaFileName;

            Semaphore sem = null;

            if (criticalSection.TryGetValue(this.ProjectConnStr, out sem) == false)
            {
                criticalSection[this.ProjectConnStr] = new Semaphore(1, 1);
            }

            // Load GME model
            this.OpenProject();

            sotInterpreter = new CyPhySoT.CyPhySoTInterpreter();
            sotInterpreter.Initialize(this.Project);
            sotInterpreter.set_ComponentParameter("run_silent", "true");
            sotInterpreter.set_ComponentParameter("do_config", "false");
            sotInterpreter.set_ComponentParameter("automation", "true");
            sotInterpreter.set_ComponentParameter("console_messages", "off");
            sotInterpreter.set_ComponentParameter("output_dir", Path.GetDirectoryName(this.WorkingDirectory));

            Type t = Type.GetTypeFromProgID("Mga.MgaFCOs");
            MgaFCOs selectedObjs = Activator.CreateInstance(t) as MgaFCOs;

            // run sot
            sotInterpreter.InvokeEx(this.Project, this.CurrentObj, selectedObjs, 0);

            // TODO: update value flow!

            this.TestBenches = sotInterpreter.TestBenches;

            // Test bench/job map
            foreach (var testbench in this.TestBenches)
            {
                Job job = new Job(this.Server);

                if (testbench.DownstreamTestBenches.Count != 0 ||
                    testbench.UpstreamTestBenches.Count != 0)
                {
                    job.RerunEnabled = false;
                }

                this.TestBenchJobMap.Add(testbench, job);
            }

            // get all testbenches from sot that can be run without dependency
            foreach (var testbench in this.TestBenches.Where(x => x.UpstreamTestBenches.Count == 0))
            {
                RunTestBenchAndStart(testbench);
            }

            this.SaveAndCloseProject();
        }

        private void MonitorAndPost(Job job)
        {
            var kvp = this.TestBenchJobMap.FirstOrDefault(x => x.Value == job);
            {
                foreach (var tbdown in kvp.Key.DownstreamTestBenches)
                {
                    if (tbdown.UpstreamTestBenches.All(x => this.TestBenchJobMap[x].Status == Job.StatusEnum.Succeeded))
                    {
                        var downJob = this.TestBenchJobMap[tbdown];
                        var jobInServer = this.Server.Jobs.FirstOrDefault(x => x.Id == downJob.Id);
                        if (jobInServer != null)
                        {
                            // already posted.
                            continue;
                        }
                        if (downJob.Status != Job.StatusEnum.Succeeded ||
                            downJob.IsFailed())
                        {
                            this.OpenProject();

                            // get all upstream test bench results back to the model.
                            tbdown.UpstreamTestBenches.ForEach(x => GetResultsBackToModel(x));
                            // propagate the values between test benches
                            PropagateValueFlow();
                            // call interpreters for this test bench and post it to the job manager
                            RunTestBenchAndStart(tbdown);

                            this.SaveAndCloseProject();
                        }
                    }
                }
            }
        }

        private void PropagateValueFlow()
        {
            //this.OpenProject();

            try
            {
                Trace.TraceInformation("BEGIN: Value flow propagation using formula evaluator");

                Type tFormulaEval = Type.GetTypeFromProgID("MGA.Interpreter.CyPhyFormulaEvaluator");
                IMgaComponentEx formulaEval = Activator.CreateInstance(tFormulaEval) as IMgaComponentEx;

                Type tMgaFCOs = Type.GetTypeFromProgID("Mga.MgaFCOs");
                MgaFCOs selectedObjs = Activator.CreateInstance(tMgaFCOs) as MgaFCOs;

                formulaEval.Initialize(this.Project);
                formulaEval.InvokeEx(this.Project, this.CurrentObj, selectedObjs, 128);
                Trace.TraceInformation("END: Value flow propagation using formula evaluator ");
            }
            catch (Exception ex)
            {
                Trace.TraceError(ex.ToString());
            }

            //this.SaveAndCloseProject();
        }

        /// <summary>
        /// Opens the project and find the CurrentObject
        /// Note: You must call SaveAndCloseProject(), which releases the taken semaphore.
        /// </summary>
        private void OpenProject()
        {
            criticalSection[this.ProjectConnStr].WaitOne();

            // Always creating a new instance of MgaProject
            Type type = Type.GetTypeFromProgID("Mga.MgaProject");
            this.Project = Activator.CreateInstance(type) as MgaProject;

            bool ro_mode;
            Trace.TraceInformation("Opening project {0}", this.ProjectConnStr);

            // Do NOT rely on ro_mode!
            this.Project.Open(this.ProjectConnStr, out ro_mode);

            try
            {
                var terr = this.Project.BeginTransactionInNewTerr();

                // ALWAYS find object 
                this.CurrentObj = this.Project.GetFCOByID(sotConfig.SoTID);
                currentObjectName = this.CurrentObj.Name;
                sotName = this.CurrentObj.RegistryValue["SoTUniqueName"];

                if (string.IsNullOrEmpty(sotName))
                {
                    sotName = currentObjectName;
                }

                this.Project.AbortTransaction();

                // FIXME: do we need this?
                //terr.Destroy();
            }
            catch (COMException ex)
            {
                // FCO was not found
                Trace.TraceError(ex.ToString());
                return;
            }
        }

        private static Dictionary<string, Semaphore> criticalSection = new Dictionary<string, Semaphore>();

        /// <summary>
        /// Saves and closes the project after this call do NOT use Project and CurrentObj
        /// Note: You must call OpenProject(), which takes a semaphore.
        /// </summary>
        private void SaveAndCloseProject()
        {
            // Other solution
            //var newname = this.ProjectConnStr + "_sottmp";
            //this.Project.Save(this.ProjectConnStr + "_sottmp");
            //this.Project.Close();
            //File.Copy(newname.Substring("MGA=".Length), this.ProjectConnStr.Substring("MGA=".Length), true);
            // this line also fails!
            //File.Delete(newname.Substring("MGA=".Length));

            //Trace.TraceInformation("Saving project {0}", this.ProjectConnStr);
            //this.Project.Save();

            Trace.TraceInformation("Saving and closing project {0}", this.ProjectConnStr);
            this.Project.Close();


            //GC.WaitForPendingFinalizers();
            //GC.Collect();
            //GC.WaitForPendingFinalizers();
            //GC.Collect();

            criticalSection[this.ProjectConnStr].Release();
        }

        private void GetResultsBackToModel(CyPhySoT.TestBench testBench)
        {
            //this.OpenProject();

            Trace.TraceInformation("GetResultsBackToModel {0}", testBench.Name);

            var filename = Path.Combine(testBench.OutputDirectory, "summary.testresults.json");
            if (File.Exists(filename) == false)
            {
                Trace.TraceError("{0} does not exist.", filename);
                this.SaveAndCloseProject();
                return;
            }

            var tbreport = Newtonsoft.Json.JsonConvert.DeserializeObject<AVM.DDP.MetaTBReport>(File.ReadAllText(filename));

            try
            {
                var terr = this.Project.BeginTransactionInNewTerr();
                foreach (var metric in tbreport.Metrics)
                {
                    try
                    {
                        if (string.IsNullOrWhiteSpace(metric.GMEID) == false)
                        {
                            var metricMgaObj = this.Project.GetFCOByID(metric.GMEID);
                            metricMgaObj.StrAttrByName["Value"] = metric.Value.ToString();
                            Trace.TraceInformation("{0} metric object was updated.", metricMgaObj.AbsPath);
                        }
                    }
                    catch (Exception ex)
                    {
                        Trace.TraceError(ex.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                this.Project.AbortTransaction();
                Trace.TraceError(ex.ToString());
            }
            finally
            {
                this.Project.CommitTransaction();
            }

            //this.SaveAndCloseProject();
        }

        private Dictionary<CyPhySoT.TestBench, Job> TestBenchJobMap = new Dictionary<CyPhySoT.TestBench, Job>();
        public System.Collections.Concurrent.BlockingCollection<Action> SoTTodo;
        private List<CyPhySoT.TestBench> TestBenches { get; set; }

        private void RunTestBenchAndStart(CyPhySoT.TestBench testbench)
        {
            //this.OpenProject();

            // Updating with the active objects
            testbench.Project = this.Project;

            testbench.OriginalProjectFileName = sotConfig.OriginalProjectFileName;

            Exception interpreterError = null;
            try
            {
                // run interpreters
                testbench.Run();
            }
            catch (Exception e)
            {
                interpreterError = e;
            }

            //this.SaveAndCloseProject();

            // TODO: catch exceptions from interpreter
            string title = string.Format("{0}__{1}", currentObjectName, testbench.Name);
            Trace.TraceInformation("Job needs to be posted {0}", title);

            // Set up job properties
            Job job = this.TestBenchJobMap[testbench];
            job.RunCommand = testbench.RunCommand;
            job.WorkingDirectory = testbench.OutputDirectory;
            job.Title = string.Format("{0}__{1}", sotName, testbench.Name);

            // artifacts/labels
            job.Labels = testbench.Labels;

            if (string.IsNullOrWhiteSpace(job.Labels))
            {
                job.Labels = Job.DefaultLabels;
            }

            job.BuildQuery = testbench.BuildQuery;

            if (string.IsNullOrWhiteSpace(job.BuildQuery))
            {
                job.BuildQuery = Job.DefaultBuildQuery;
            }

            if (string.IsNullOrWhiteSpace(testbench.ResultsZip) == false)
            {
                job.ResultsZip = testbench.ResultsZip;
            }

            job.JobStatusChanged += JobStatusChanged;

            job.Status = Job.StatusEnum.WaitingForStart;
            this.Server.AddJob(job);

            if (interpreterError != null)
            {
                string failed_txt = Path.Combine(testbench.OutputDirectory, "_FAILED.txt");
                if (File.Exists(failed_txt) == false)
                {
                    File.WriteAllText(failed_txt, String.Format("Interpreter {0} failed: {1}", testbench.Interpreter.ComponentProgID, interpreterError.ToString()));
                }
                job.Status = Job.StatusEnum.Failed;
            }
            else
            {
                if (this.JobAction != null)
                {
                    this.JobAction(job);
                }

                job.Start();
            }

        }

        public void JobStatusChanged(Job job)
        {
            SoTTodo.Add(delegate { JobStatusChangedSTA(job); });
        }

        public void JobStatusChangedSTA(Job job)
        {
            if (job.IsFailed())
            {
                // TODO: how to indicate that the jobs are failed?
                // TODO: get all downstream jobs and mark them as failed?

                if (this.TestBenchJobMap.All(x => x.Value.Status == Job.StatusEnum.Succeeded || x.Value.IsFailed()))
                {
                    //this.Project.Close();
                }
            }
            else if (job.Status == Job.StatusEnum.Succeeded)
            {
                // FIXME: could this fail??? would be nice to put it into a try catch.
                MonitorAndPost(job);

                if (this.TestBenchJobMap.All(x => x.Value.Status == Job.StatusEnum.Succeeded || x.Value.IsFailed()))
                {
                    //this.Project.Close();
                }
            }
        }
    }
}
