import os
import json
import random
import datetime
import logging
from py_modelica.exception_classes import ModelicaInstantiationError

from abc import ABCMeta, abstractmethod

class ToolBase:
    __metaclass__ = ABCMeta

    tool_name = ''
    tool_version = ''
    tool_version_nbr = ''

    model_config = None

    status = {}
    date_time = ''

    ## instance variables
    tool_path = ''              # path to the bin folder of the tool

    model_file_name = ''        # file that needs to be loaded
    model_name = ''             # name of the model in the loaded packages

    msl_version = ''         # version of Modelica Standard Library

    mos_file_name = ''          # modelica script files for compiling the model

    result_mat = ''             # contains the latest simulation results
    base_result_mat = ''        # contains the expected simulation results

    working_dir = ''            # contains the temporary files and executables
    root_dir = ''
    mo_dir = ''                 # contains the modelica file, (package or model)

    output_dir = ''             # relative or absolute

    variable_filter = []        # list of names of variables to save/load to/from mat-file

    experiment = {}             # dictionary with StartTime, StopTime, Tolerance,
                                # NumberOfIntervals, Interval and Algorithm.

    model_is_compiled = False   # flag for telling if the model was compiled
    model_did_simulate = False  # flag for telling if the model has been simulated

    generate_status = True      # determines if the status will be updated and saved during run.

    lib_package_paths = []      # paths to additional packages
    lib_package_names = []      # names of additional packages

    compilation_time = -1
    translation_time = -1
    make_time = -1
    simulation_time = -1
    total_time = -1

    def _initialize(self,
                    model_config):
        """
        Creates a new instance of a modelica simulation.

        dictionary : model_config

            Mandatory Keys : 'model_name' (str), 'model_file_name' (str)

            Optional Keys : 'MSL_version' (str), 'variable_filter' ([str]),
                            'result_file' (str), 'experiment' ({str})
        """

        print ' --- =====  See debug.log for error/debug messages ===== --- \n'
        print ' in {0}'.format(os.getcwd())

        # create a logger, (will only be written to if no other logger defined 'higher' up)
        logging.basicConfig(filename="debug.log",
            format="%(asctime)s %(levelname)s: %(message)s",
            datefmt="%Y-%m-%d %H:%M:%S")

        log = logging.getLogger()
        # always use highest level of debugging
        log.setLevel(logging.DEBUG)

        log.debug(" --- ==== ******************************* ==== ---")
        log.info(" --- ==== ******* New Run Started ******* ==== ---")
        self.date_time = '{0}'.format(datetime.datetime.today())
        log.debug(" --- ==== * {0} ** ==== ---".format(self.date_time))
        log.debug(" --- ==== ******************************* ==== ---")
        log.debug("Entered _initialize")
        log.info("tool_name : {0}".format(self.tool_name))
        log.info("tool_path : {0}".format(self.tool_path))
        self.root_dir = os.getcwd()

        self.model_config = model_config

        # Mandatory keys in dictionary
        try:
            model_file_name = self.model_config['model_file_name']
            if model_file_name == "":
                self.model_file_name = ""
                log.info("No model_file name given, assumes model is in Modelica Standard Library")
            else:
                self.model_file_name = os.path.normpath(os.path.join(os.getcwd(), model_file_name))
                self.mo_dir = os.path.dirname(self.model_file_name)
                log.info("mo_dir : {}".format(self.mo_dir))
                log.info("model_file_name : {0}".format(self.model_file_name))

            model_name = self.model_config['model_name']
            if model_name == "":
                base_name = os.path.basename(model_file_name)
                self.model_name = os.path.splitext(base_name)[0]
                log.info("No model_name given, uses model_file_name without .mo")
            else:
                self.model_name = model_name
                log.info("model_name : {0}".format(self.model_name))
        except KeyError as err:
            raise ModelicaInstantiationError("Mandatory key missing in model_config : {0}".format(err.message))

        # optional keys in dictionary
        if model_config.has_key('MSL_version'):
            self.msl_version = self.model_config['MSL_version']
        else:
            self.msl_version = "3.2"

        if model_config.has_key('variable_filter'):
            self.variable_filter = self.model_config['variable_filter']

        if model_config.has_key('result_file'):
            result_file = self.model_config['result_file']
        else:
            result_file = ""
        if result_file == "":
            log.debug("No result file specified, no comparison of simulation result will be made")
        else:
            self.base_result_mat = os.path.normpath(os.path.join(os.getcwd(),result_file))
            log.info("base_result_mat : {0}".format(self.base_result_mat))
            if not os.path.exists(self.base_result_mat):
                log.warning("Given result_file {0} does not exist".format(self.base_result_mat))
                self.base_result_mat = ""
                log.warning('base_result_mat set to "",')
                log.warning("no comparison will be made.")

        if model_config.has_key('experiment'):
            self.experiment = dict(
                StartTime = model_config['experiment']['StartTime'],
                StopTime = model_config['experiment']['StopTime'],
                NumberOfIntervals = model_config['experiment']['NumberOfIntervals'],
                Tolerance = model_config['experiment']['Tolerance'])
            # Algorithm
            if model_config['experiment'].has_key('Algorithm'):
                if self.tool_name.startswith('Dymola'):
                    self.experiment.update({'Algorithm':
                        self.model_config['experiment']['Algorithm']['Dymola']})
                elif self.tool_name is 'OpenModelica':
                    self.experiment.update({'Algorithm':
                        self.model_config['experiment']['Algorithm']['OpenModelica']})
            else: # py_modelica 12.09
                self.experiment.update({'Algorithm':'dassl'})
            # Interval
            if model_config['experiment'].has_key('IntervalMethod'):
                if model_config['experiment']['IntervalMethod'] == 'Interval':
                    self.experiment.update({"NumberOfIntervals": "0"})
                    self.experiment.update({"Interval": model_config['experiment']['Interval']})
                else:
                    self.experiment.update({"NumberOfIntervals":
                        model_config['experiment']['NumberOfIntervals']})
                    self.experiment.update({"Interval": "0"})
            else: # py_modelica 12.09
                self.experiment.update({"NumberOfIntervals":
                    model_config['experiment']['NumberOfIntervals']})
                self.experiment.update({"Interval": "0"})
        else:
            self.experiment = dict(StartTime='0',
                                   StopTime='1',
                                   NumberOfIntervals='500',
                                   Interval = '0',
                                   Tolerance='1e-5',
                                   Algorithm='dassl')
            log.info("No experiment data given, default values will be used...")
        log.info("Experiment settings : {0}".format(self.experiment))

        if model_config.has_key('lib_package_paths'):
            for lib_path in self.model_config['lib_package_paths']:
                if lib_path:
                    self.lib_package_paths.append(str(lib_path))

        if model_config.has_key('lib_package_names'):
            for lib_name in self.model_config['lib_package_names']:
                if lib_name:
                    self.lib_package_names.append(lib_name)



    # end of __initialize__

    @abstractmethod
    def compile_model(self):
        return bool

    @abstractmethod
    def simulate_model(self):
        return bool

    @abstractmethod
    def change_experiment(self,
                          start_time='0',
                          stop_time='1',
                          increment='',
                          n_interval='500',
                          tolerance='1e-5',
                          max_fixed_step='',
                          solver='dassl',
                          output_format='',
                          variable_filter=''):
        return bool

    @abstractmethod
    def change_parameter(self, change_dict):
        return bool

    def compare_results(self):
        results = {}
        results['base_mat_exists'] = False
        results['binary_comparison'] = False
        results['simulation_time_interval_same'] = False
        results['end_values_match'] = False
        results['unmatched_parameters'] = []

        return results

    def _generate_new_status(self):
        """
        Generates dictionary containing status from the run

        """
        log = logging.getLogger()
        log.debug("Entered _generate_status")

        self.random_md5 = hex(random.getrandbits(128))[2:-1]
        self.status.update({self.random_md5:{}})

        self.status[self.random_md5]['datetime'] = self.date_time
        self.status[self.random_md5]['model'] = self.model_config
        self.status[self.random_md5]['tool'] = {'tool_name' : self.tool_name}
        if self.tool_name == "OpenModelica":
            self.status[self.random_md5]['tool']['version'] = self.om_version
        self.status[self.random_md5]['experiment'] = self.experiment

        execution_statistics = {'compilation_time': self.compilation_time,
                                'simulation_time': self.simulation_time,
                                'total_time': self.total_time}

        self.status[self.random_md5]['execution_statistics'] = execution_statistics

        if self.base_result_mat:
            results = self.compare_results()
        else:
            results = None

        self.status[self.random_md5]['results'] = results

    def save_status(self, file_name="report.mo.json"):
        """
        Saves/updates current status at given file_name.

        """

        log = logging.getLogger()
        log.debug("Entered save_status")
        status_file_name = os.path.join(self.root_dir, file_name)

        self._generate_new_status()

        existing_status = {}
        if os.path.exists(status_file_name):
            log.info("Found existing status {0}".format(file_name))
            with open(status_file_name, 'r') as status_file:
                existing_status = json.load(status_file)
        self.status.update(existing_status)

        with open(status_file_name, 'w') as status_file:
            try:
                json.dump(self.status, status_file, indent = 4)
                log.info("Json has been dumped at {0}".format(status_file_name))
            except:
                log.error("Could not dump to {0} file properly!".format(status_file_name))
        os.chdir(self.root_dir)
    # end of save_status
