"""
Functions called after or before simulations of CyPhy-generated modelica models.
If any of these fail, they will throw an IOProcessingError
"""

import os
import json
from numpy import NaN

from py_modelica.mat_file_functions import MatFile2Dict

REPORT_FILE = 'summary.testresults.json'

def _except_NaN(x):
    print "Found invalid {0}".format(x)
    if x == 'NaN':
        return NaN

def _get_maximum_value(time_series):
    """
    Converts the list time_series into numpy array
    and returns the maximum value.
    """
    import numpy as np

    np_time_series = np.array(time_series)
    max_value = np_time_series.max()

    return max_value

def get_analysis_status(report_file=REPORT_FILE):
    """
    Reads and returns the analysis status from report_file,
    default return is 'FAILED'

    """
    if not report_file:
        return

    with open(report_file,'r') as file_in:
        result_json = json.load(file_in, parse_constant=_except_NaN)

    analysis_status = 'FAILED'

    if 'AnalysisStatus' in result_json:
        analysis_status = result_json['AnalysisStatus']

    return analysis_status

def update_analysis_status(new_status='', msg='', report_file=REPORT_FILE):
    """
    Updates the AnalysisStatus value in the report_file

    new_status has to be within {'UNEXECUTED', 'OK', 'FAILED'}

    """
    if not report_file:
        return

    with open(report_file,'r') as file_in:
        result_json = json.load(file_in, parse_constant=_except_NaN)

    if 'AnalysisStatus' in result_json:
        result_json['AnalysisStatus']= new_status

    if new_status == 'FAILED':
        msg_dic = {'ErrorMessage' : msg}
        result_json.update(msg_dic)

    with open(report_file,'wb') as file_out:
        json.dump(result_json, file_out, indent=4)

def update_parameters_in_report_json(params, report_file=REPORT_FILE):
    """
    Updates the parameter values in report_file (json),
    using the params

    """
    if not report_file:
        return

    # read current summary report, which contains the metrics
    with open(report_file,'r') as file_in:
        result_json = json.load(file_in, parse_constant=_except_NaN)

    if 'Parameters' in result_json:
        for paramOld in result_json['Parameters']:
            if 'Name' in paramOld and 'Value' in paramOld:
                if paramOld['Name'] in params:
                    paramOld['Value'] = str(params[paramOld['Name']])
                else:
                    # parameter was not found in results
                    pass
            else:
                # create warning message
                pass
                # update json file with the new values
        with open(report_file,'wb') as file_out:
            json.dump(result_json, file_out, indent=4)
    else:
        # create warning message
        pass

def update_metrics_in_report_json(mat_file='', filter='filter.json', report_file=REPORT_FILE, max_value=False):
    """
    Update metrics in report_file after simulation,
    max_value=False -> using the last value in the mat_file
    max_value=True -> Updates metrics given in filter (N.B. pass list in this case) with max value.

    filter can be either a .json file, a list of strings or ''.
        if '' -> all variables in the mat_file will be considered.

    """

    if not os.path.exists(report_file):
        raise IOError ('Report file does not exits : {0}'.format(report_file))

    if not os.path.exists(mat_file):
        msg = 'Given result file does not exist: {0}'.format(mat_file)
        raise IOError(msg)

    if isinstance(filter, str):
        if os.path.exists(filter):
            with open(filter,'r') as file_in:
                filter = json.load(file_in)
        else:
            print 'The given filter file : {0}, does not exist!'.format(filter)
            print 'Result mat-file will be loaded without filtering...'
            filter = []

    # file exists
    mat_converter = MatFile2Dict(mat_file, filter, False)
    result = mat_converter.get_results()

    # read current summary report, which contains the metrics
    with open(report_file,'r') as file_in:
        result_json = json.load(file_in, parse_constant=_except_NaN)

    if 'Metrics' in result_json:
        for metric in result_json['Metrics']:
            if 'Name' in metric and 'Value' in metric:
                if metric['Name'] in result:
                    if max_value:
                        metric['Value'] = str(_get_maximum_value(result[metric['Name']]))
                    else:
                        metric['Value'] = str(result[metric['Name']][-1])
                else:
                    if not max_value:
                        # only print when regular metric update
                        print 'Metric : {0} was not found in results'.format(metric['Name'])
                    pass
            else:
                print 'Metric item : {0} does not have right format'.format(metric)
                pass
        # update json file with the new values
        with open(report_file,'wb') as file_out:
            json.dump(result_json, file_out, indent=4)
    else:
        print 'Report file {0} does not have any Metrics defined..'
        pass

def get_parameters_from_report_json(report_file=REPORT_FILE):
    """
    Reads in parameters from report_file
    """
    if not report_file:
        return

    parameters = {}
    with open(report_file) as file_in:
        result_json = json.load(file_in, parse_constant=_except_NaN)

    if result_json.has_key('Parameters'):
        for param in result_json['Parameters']:
            try:
                parameters.update({str(param['Name']):float(param['Value'])})
            except ValueError:
                # if not float assume it's a boolean, thus string
                parameters.update({str(param['Name']):str(param['Value'])})

    return parameters

def get_metrics_from_report_json(report_file=REPORT_FILE):
    """
    Reads the values of the metrics from the report_file
    and returns them as a dictionary. (Values are strings.)

    """

    if not report_file:
        return

    metric_dict = {}
    # read current json file with
    with open(report_file,'r') as file_in:
        result_json = json.load(file_in, parse_constant=_except_NaN)

    if 'Metrics' in result_json:
        for metric in result_json['Metrics']:
            if 'Name' in metric and 'Value' in metric:
                metric_dict.update({metric['Name']:metric['Value']})
            else:
                # create warning message
                pass
    else:
        # create warning message
        pass
    return metric_dict

def update_metrics_and_parameters(metrics, parameters, report_file=REPORT_FILE):
    """
    f
    """
    if not report_file:
        return

    with open(report_file) as file_in:
        rep_json = json.load(file_in, parse_constant=_except_NaN)

    rep_json['Parameters'] = parameters
    rep_json['Metrics'] = metrics

    with open(report_file,'wb') as file_out:
        json.dump(rep_json, file_out, indent=4)

def copy_metrics_and_parameters(report_file=REPORT_FILE):
    """
    f
    """
    if not report_file:
        return

    metrics = {}
    parameters = {}
    with open(report_file) as file_in:
        rep_json = json.load(file_in, parse_constant=_except_NaN)

    if 'Metrics' in rep_json:
        metrics = rep_json['Metrics']

    if 'Parameters' in rep_json:
        parameters = rep_json['Parameters']

    return metrics, parameters

def check_limits(mat_file='', limits_file='limits.json'):
    """
    Check if limits are withheld.

    """

    if not os.path.exists(mat_file):
        print 'Given result file does not exist: {0}'.format(mat_file)

    # read current limit file, which contains the metrics
    with open(limits_file,'r') as file_in:
        limits_json = json.load(file_in, parse_constant=_except_NaN)

    if 'LimitChecks' in limits_json:
        # only load the variables in limits
        filter = []
        for checkval in limits_json['LimitChecks']:
            filter.append(checkval['VariableFullPath'])

        mat_converter = MatFile2Dict(mat_file, filter, False)
        result = mat_converter.get_results()

        for checkval in limits_json['LimitChecks']:
            if 'VariableFullPath' in checkval and \
               'Maximum' in checkval['Attributes'] and\
               'Minimum' in checkval['Attributes']:

                print checkval['VariableFullPath']
                print checkval['VariableName']
                print checkval['LimitName']
                print checkval['Attributes']
                min = checkval['Attributes']['Minimum']
                max = checkval['Attributes']['Maximum']
                print min
                print max
                if checkval['LimitName'] in result:
                #if 'MaxSpeed' in result_mat[0]:
                    val = float(str(result[checkval['LimitName']][-1]))
                    if val < min or val > max:
                        checkval['LimitExceeded'] = 'true'
                        print 'Limit Exceeded'
                    else:
                        checkval['LimitExceeded'] = 'false'
                        print 'Signal within limit'
                else:
                    # metric was not found in results
                    pass
            else:
                # create warning message
                pass
            # update json file with the new values
        with open(limits_file,'wb') as file_out:
            json.dump(limits_json, file_out)

    else:
        # create warning message
        pass