/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Runtime.InteropServices;
using GME.MGA;
using GME;
using GME.MGA.Parser;
using System.IO;
using System.Diagnostics;

namespace ESMoL_toolchain
{
    [ComVisible(true),
    Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    public class Run_ESMoL_toolchain : GME.MGA.IMgaComponentEx, GME.MGA.Core.IGMEVersionInfo
    {
        private string getPath(IMgaFCO fco)
        {
            string path = fco.Name;
            while (fco.ParentModel != null)
            {
                path = fco.ParentModel.Name + "/" + path;
                fco = fco.ParentModel;
            }
            IMgaFolder folder = fco.ParentFolder;
            while (folder != null)
            {
                path = folder.Name + "/" + path;
                folder = folder.ParentFolder;
            }
            return "/" + path;
        }

        public void InvokeEx2(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            string projectPath = Path.GetDirectoryName(project.ProjectConnStr.Substring(4)); // skip mga=
            string projectName = Path.GetFileNameWithoutExtension(project.ProjectConnStr.Substring(4));
            string esmolFilename = project.ProjectConnStr.Substring(4);
            baseOutputDir = Path.Combine(projectPath, projectName + "_ESMoL");
            GME.CSharp.GMEConsole console = GME.CSharp.GMEConsole.CreateFromProject(project);
            console.Out.WriteLine("Output directory is " + baseOutputDir);

            VCP_PATH = Environment.GetEnvironmentVariable("VCP_PATH");
            if (VCP_PATH == null || VCP_PATH == "")
            {
                throw new ApplicationException("Environment variable VCP_PATH doesn't exist. Please install the ESMoL toolchain and restart GME.");
            }
            if (!Directory.Exists(VCP_PATH))
            {
                throw new ApplicationException("VCP_PATH='" + VCP_PATH + "' doesn't exist. Please install the ESMoL toolchain and restart GME.");
            }
            if (!Directory.Exists(projectPath))
            {
                throw new ApplicationException(projectPath + " doesn't exist");
            }
            ensureDir(baseOutputDir);
            ensureDir(getSubdir("SL_CodeGen"));
            ensureDir(getSubdir("SF_CodeGen"));
            ensureDir(getSubdir("C_CodeGen"));
            ensureDir(getSubdir("stage1"));
            ensureDir(getSubdir("stage2"));
            ensureDir(getSubdir("target_truetime"));

            runProgram("SL_CodeGen.exe", new string[] { "-p", getSubdir("SL_Codegen"), esmolFilename });
            runProgram("C_CodeGen.exe", new string[] { "-p", getSubdir("C_CodeGen"), "-f", esmolFilename });
            runProgram("SF_CodeGen.exe", new string[] { "-t", getSubdir("SF_CodeGen"), esmolFilename });
            string stage1File = getSubdir("stage1") + "\\" + projectName + ".xml";
            runProgram("Stage1", new string[] { "-f", esmolFilename, "-o", stage1File });
            string schedFile = getSubdir("stage2") + "\\sched.scs";
            runProgram("Stage2", new string[] { "-f", stage1File, "-o", schedFile, "-t", "Sched", "-r", "1ms" });
            string rsltFile = getSubdir("stage2") + "\\results.rslt";
            runProgram("SchedTool", new string[] { "-f", schedFile, "-o", rsltFile });
            runProgram("SchedResults", new string[] { "-r", rsltFile, "-e", esmolFilename, "-a", stage1File });
            runProgram("Stage2", new string[] { "-t", "TrueTime", "-f", stage1File, "-p", getSubdir("target_truetime"), "-o", projectName, "-m" });

            foreach (string[] s in new string[][] { Directory.GetFiles(getSubdir("SF_CodeGen")), Directory.GetFiles(getSubdir("SL_CodeGen")), Directory.GetFiles(getSubdir("C_CodeGen")) })
            {
                foreach (string file in s)
                {
                    File.Copy(file, Path.Combine(getSubdir("target_truetime"), Path.GetFileName(file)), true);
                }
            }
        }
        string baseOutputDir;
        string VCP_PATH;
        private string getSubdir(string subdir)
        {
            return baseOutputDir + "\\" + subdir;
        }

        private void ensureDir(string subdir)
        {
            if (!Directory.Exists(subdir))
            {
                Directory.CreateDirectory(subdir);
            }
        }

        private string escapeArguments(string[] arguments)
        {
            string ret = "";
            foreach (string arg in arguments)
            {
                ret += "\"" + arg + "\" ";
            }
            return ret;
        }
        private void runProgram(string program, string[] arguments)
        {
            runProgram(program, escapeArguments(arguments));
        }

        private void runProgram(string program, string arguments)
        {
            program = VCP_PATH + "\\bin\\" + program;
            Process p = new Process();
            // Redirect the output stream of the child process.
            p.StartInfo.UseShellExecute = false;
            p.StartInfo.CreateNoWindow = true;
            p.StartInfo.RedirectStandardOutput = true;
            p.StartInfo.RedirectStandardError = false;
            p.StartInfo.FileName = program;
            p.StartInfo.Arguments = arguments;
            p.Start();
            string output = p.StandardOutput.ReadToEnd();
            p.WaitForExit();
            if (p.ExitCode != 0)
            {
                p = new Process();
                p.StartInfo.FileName = "cmd.exe";
                p.StartInfo.Arguments = " /k \"\"" + program + "\" " + arguments + "\"";
                p.Start();

                throw new ApplicationException("Running " + program + " " + arguments + " failed: " + output);
            }
        }

        private delegate void voidDelegate();

        private void performInROTransaction(IMgaProject project, voidDelegate d)
        {
            IMgaTerritory territory = project.BeginTransactionInNewTerr(transactiontype_enum.TRANSACTION_READ_ONLY);
            try
            {
                d();
            }
            finally
            {
                try
                {
                    project.AbortTransaction();
                }
                catch { }
                try
                {
                    territory.Destroy();
                }
                catch { }
            }

        }

        #region IMgaComponentEx Members

        public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            try
            {
                InvokeEx2(project, currentobj, selectedobjs, param);
            }
            finally
            {
                project = null;
                currentobj = null;
                selectedobjs = null;
                GC.Collect();
                GC.WaitForPendingFinalizers();
            }
        }

        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                RegistrationServices regService = new RegistrationServices();
                return regService.GetProgIdForType(GetType());
            }
        }

        public componenttype_enum ComponentType
        {
            get { return GME.MGA.componenttype_enum.COMPONENTTYPE_INTERPRETER; }
        }

        public void Enable(bool newval)
        {
        }

        public void Initialize(MgaProject p)
        {
        }

        protected bool interactiveMode = false;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }

        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            InvokeEx(Project, null, selectedobjs, 0);
        }

        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
        }

        #endregion

        #region IGMEVersionInfo Members

        public GME.MGA.Core.GMEInterfaceVersion_enum version
        {
            get { return GME.MGA.Core.GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();
        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnregister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }
    }

}
