/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef _UDM_COMPARATOR_
#define _UDM_COMPARATOR_

#include <iostream>
#include <set>
#include <map>
#include <boost/lexical_cast.hpp>

#include "UdmBase.h"
#include "Uml.h"

class UdmComparator {
public:
	typedef std::map< std::string, Uml::Attribute > NameUmlAttributeMap;
	typedef std::set< Uml::Attribute > UmlAttributeSet;
	typedef std::set< Uml::AssociationRole > UmlAssociationRoleSet;

private:
	static NameUmlAttributeMap getNameUmlAttributeMap( Uml::Class umlClass );
	static UmlAttributeSet getAllUmlAttributes( Uml::Class umlClass );
	static UmlAssociationRoleSet getAllUmlAssociationRoles( Uml::Class umlClass );

	class ObjectName {
	public:
		typedef map< Uml::Class, Uml::Attribute > UmlClassNameAttributeMap;

	private:
		UmlClassNameAttributeMap _umlClassNameAttributeMap;
	
	public:
		std::string operator()( Udm::Object udmObject );
	};

public:
	class Report {
	public:
		typedef std::map< Udm::Object, int > UdmObjectDifferenceMap;

	private:
		std::string _reportString;
		UdmObjectDifferenceMap _objectDifferences;

	public:
		std::string &getString( void ) {
			return _reportString;
		}

		void setString( const std::string &reportString ) {
			_reportString = reportString;
		}

		void addToReport( Udm::Object udmObject ) {
			int differences = 0;
			UdmObjectDifferenceMap::iterator udmItr = _objectDifferences.find( udmObject );
			if ( udmItr != _objectDifferences.end() ) differences = udmItr->second;
			if ( differences <= 0 ) return;

			std::string addString;

			for( int ix = 0 ; ix < udmObject.depth_level() ; ++ix ) addString += "    ";

			addString += udmObject.getPath( "/" ) + ": " + boost::lexical_cast< std::string >( differences ) + "\n";
			_reportString = addString + _reportString;
		}

		void incrementDifferences( Udm::Object udmObject ) {
			UdmObjectDifferenceMap::iterator udmItr = _objectDifferences.find( udmObject );
			if ( udmItr == _objectDifferences.end() ) udmItr = _objectDifferences.insert(  std::make_pair( udmObject, 0 )  ).first;
			++(udmItr->second);
		}

		int getDifferences( Udm::Object udmObject ) {
			UdmObjectDifferenceMap::iterator udmItr = _objectDifferences.find( udmObject );
			if ( udmItr == _objectDifferences.end() ) udmItr = _objectDifferences.insert(  std::make_pair( udmObject, 0 )  ).first;
			return udmItr->second;
		}

		void addToDifferences( Udm::Object udmChildObject ) {
			UdmObjectDifferenceMap::iterator childUDMItr = _objectDifferences.find( udmChildObject );
			if ( childUDMItr == _objectDifferences.end() ) return;

			Udm::Object udmParentObject = udmChildObject.GetParent();
			UdmObjectDifferenceMap::iterator udmItr = _objectDifferences.find( udmParentObject );
			if ( udmItr == _objectDifferences.end() ) udmItr = _objectDifferences.insert(  std::make_pair( udmParentObject, 0 )  ).first;
			udmItr->second += childUDMItr->second;
		}

		static Report &get_singleton( void ) {
			static Report report;
			return report;
		}
	};

	static ObjectName &getObjectNameSingleton( void ) {
		static ObjectName objectName;
		return objectName;
	}

	struct ChildObjectComparator {
		bool operator()( const Udm::Object &udmObject1, const Udm::Object &udmObject2 );
	};
	struct AssociationObjectComparator {
		bool operator()( const Udm::Object &udmObject1, const Udm::Object &udmObject2 );
	};

	typedef set< Udm::Object > UdmObjectSet;
	typedef set< Udm::Object, ChildObjectComparator > UdmChildObjectSet;
	typedef set< Udm::Object, AssociationObjectComparator > UdmAssociationObjectSet;

	typedef set< Uml::Class > UmlClassSet;

	typedef std::set< std::string > StringSet;
	typedef std::map< std::string, StringSet > StringStringSetMap;

	class ClassNameFilter {
	private:
		StringSet _inclusiveClassNameSet;
		StringSet _exclusiveClassNameSet;

		StringSet _includedClassNameSet;
		StringSet _excludedClassNameSet;

		StringStringSetMap _inclusiveClassNameAttributeNameSetMap;
		StringStringSetMap _exclusiveClassNameAttributeNameSetMap;

	public:
		void setInclusiveClassNameSet( const StringSet &inclusiveClassNameSet ) {
			_inclusiveClassNameSet = inclusiveClassNameSet;
		}
		StringSet getInclusiveClassNameSet( void ) {
			return _inclusiveClassNameSet;
		}

		StringSet getIncludedClassNameSet( void ) {
			return _includedClassNameSet;
		}

		void setExclusiveClassNameSet( const StringSet &exclusiveClassNameSet ) {
			_exclusiveClassNameSet = exclusiveClassNameSet;
		}
		StringSet getExclusiveClassNameSet( void ) {
			return _exclusiveClassNameSet;
		}

		StringSet getExcludedClassNameSet( void ) {
			return _excludedClassNameSet;
		}

		void setInclusiveClassNameAttributeNameMap( const StringStringSetMap &inclusiveClassNameAttributeNameMap ) {
			_inclusiveClassNameAttributeNameSetMap = inclusiveClassNameAttributeNameMap;
		}

		void setExclusiveClassNameAttributeNameMap( const StringStringSetMap &exclusiveClassNameAttributeNameMap ) {
			_exclusiveClassNameAttributeNameSetMap = exclusiveClassNameAttributeNameMap;
		}

		UdmObjectSet filterUdmObjectSet( const UdmObjectSet &udmObjectSet );
		UdmObjectSet filterConnections( const UdmObjectSet &udmObjectSet );

		UmlAttributeSet filterUmlAttributeSet( Uml::Class umlClass );
	};

	typedef map< Udm::Object, Udm::Object > UdmObjectMap;

	typedef set< Uml::CompositionChildRole > CompositionChildRoleSet;


private:
	UdmObjectMap _udmObjectMap;
	ClassNameFilter _classNameFilter;

public:
	void setClassNameFilter( const ClassNameFilter &classNameFilter ) {
		_classNameFilter = classNameFilter;
	}

	bool compareNode( Udm::Object udmObject1, Udm::Object udmObject2 );

private:
	bool compareNodeAux( Udm::Object udmObject1, Udm::Object udmObject2 );

	typedef std::pair< int, int > IntPair;
	typedef std::map< std::string, IntPair > ChildrenClassesTallyMap;

	void tallyChildrenClasses( const UdmObjectSet &udmObjectSet1, const UdmObjectSet &udmObjectSet2 );
};

#endif