/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GME.MGA;
using GME.Util;
using System.Runtime.InteropServices;

namespace META
{
    public class ComComponent
    {
        #region Public Static Properties

        /// <summary>
        /// Stores common information about the project.
        /// InvokeExSimple will call the InvokeEx function using these parameters.
        /// </summary>
        public struct Options
        {
            public static MgaFCOs SelectedObjects = (MgaFCOs)Activator.CreateInstance(Type.GetTypeFromProgID("Mga.MgaFCOs"));
            public static int Param = 128;
        };

        /// <summary>
        /// Common component parameters. Same for each component.
        /// These Key,Value pairs will be sent to
        /// the MgaComponent after the initialization and before the 
        /// component will be called.
        /// </summary>
        public static Dictionary<string, string> CommonParameters =
            new Dictionary<string, string>();

        #endregion

        #region Public Properties

        /// <summary>
        /// Name of this component.
        /// </summary>
        public string ProgId { get; set; }

        public List<string> ParameterList
        {
            get
            {
                if (MgaComponent != null)
                {
                    string[] _parameterlist = MgaComponent.ComponentParameter["parameter_list"] as string[];
                    if (_parameterlist != null)
                    {
                        return _parameterlist.ToList();
                    }
                    return new List<string>();
                }
                return new List<string>();
            }
        }
        public string GetParameterValue(string parameter)
        {
            if (MgaComponent != null)
            {
                try
                {
                    object value = MgaComponent.ComponentParameter[parameter];
                    if (value is string)
                        return value as string;
                }
                catch (COMException)
                {
                    return null;
                }
            }
            return null;
        }

        public string Name { get; set; }
        public string Description { get; set; }
        public string Help { get; set; }

        /// <summary>
        /// Component parameters. These Key,Value pairs will be sent to
        /// the MgaComponent after the initialization and before the 
        /// component will be called.
        /// </summary>
        public Dictionary<string, string> Parameters { get; set; }

        /// <summary>
        /// True if the interpreter requires a current object.
        /// </summary>
        public bool CurrentObjectRequired { get; set; }

        /// <summary>
        /// Mga COM component wrapper.
        /// </summary>
        public IMgaComponentEx MgaComponent { get; set; }

        /// <summary>
        /// ToolTip of the component.
        /// </summary>
        public string ToolTip
        {
            get
            {
                MgaRegistrar registrar = new MgaRegistrar();
                string desc = "";
                try
                {
                    // FIXME: if the component is a C# dll this line throws a
                    // COMException.
                    desc = registrar.get_ComponentExtraInfo(
                        regaccessmode_enum.REGACCESS_BOTH,
                        ProgId,
                        "ToolTip");
                }
                catch (COMException)
                {
                    try
                    {
                        componenttype_enum Type;
                        registrar.QueryComponent(
                            ProgId,
                            out Type,
                            out desc,
                            regaccessmode_enum.REGACCESS_BOTH);
                    }
                    catch (COMException)
                    {
                        // ignore it
                        // if the dll path is empty
                    }
                }
                if (string.IsNullOrEmpty(desc))
                {
                    return ProgId;
                }
                else
                {
                    return desc;
                }
            }
        }

        /// <summary>
        /// Type of this COM component.
        /// </summary>
        public Type Type { get; set; }

        /// <summary>
        /// True if the component can be called throw COM.
        /// </summary>
        public bool isValid { get; set; }

        /// <summary>
        /// Indicates an Error message in a string format
        /// </summary>
        public string Error
        {
            get { return isValid ? "" : "Com object was not found."; }
        }

        #endregion

        #region Public Functions
        public void Initialize(MgaProject Project)
        {
            if (isValid == false)
            {
                return;
            }

            MgaComponent.Enable(true);
            MgaComponent.Initialize(Project);
            MgaComponent.InteractiveMode = true;

            // set common paramters
            foreach (KeyValuePair<string, string> kvp in CommonParameters)
            {
                MgaComponent.ComponentParameter[kvp.Key] = kvp.Value;
            }

            // set interpreter specific paramters
            foreach (KeyValuePair<string, string> kvp in Parameters)
            {
                MgaComponent.ComponentParameter[kvp.Key] = kvp.Value;
            }
        }

        /// <summary>
        /// Calls the COM component's InvokeEx method's.
        /// Please call the initialize before you call this method.
        /// </summary>
        /// <param name="project"></param>
        /// <param name="currentObject"></param>
        /// <param name="selectedObjects"></param>
        /// <param name="param"></param>
        public void InvokeEx(
            MgaProject project,
            MgaFCO currentObject,
            MgaFCOs selectedObjects,
            int param)
        {
            if (CurrentObjectRequired == true &&
                    currentObject == null)
            {
                throw new Exception("Current object is not set for COM object.");
            }

            MgaComponent.InvokeEx(
                project,
                currentObject,
                selectedObjects,
                param);
        }

        #endregion

        public override string ToString()
        {
            if (string.IsNullOrEmpty(Name))
            {
                return ProgId;
            }
            else
            {
                return ToolTip;
                //return Name;
            }
            //return base.ToString();
        }

        #region Constructors

        public ComComponent(string name)
            : this(name, true)
        {
        }

        public ComComponent(
            string progId,
            bool currentObjectRequired)
        {
            //MGALib.IMgaProject project;
            //bool mode;
            //project.Open("MGA=...mga", out mode);
            //project.GetFCOByID("id-");
            //run_interpreter;
            //project.Close(true);

            Parameters = new Dictionary<string, string>();
            CurrentObjectRequired = currentObjectRequired;
            ProgId = progId;

            // default values
            Name = ProgId;
            Description = "Description: " + ProgId;
            Help = "Help: " + ProgId;


            // if (!ProgId.Equals("MGA.Interpreter.CyPhyDynamics", StringComparison.InvariantCultureIgnoreCase))
            //if (!ProgId.Equals(Definitions.COM[Definitions.Interpreter_enum.Dynamics], StringComparison.InvariantCultureIgnoreCase))
            //{
            //  return;
            //}

            if (string.IsNullOrWhiteSpace(ProgId))
            {
                isValid = false;
                return;
            }

            Type = Type.GetTypeFromProgID(ProgId);

            if (Type != null)
            {
                isValid = true;
                try
                {
                    MgaComponent = Activator.CreateInstance(Type) as IMgaComponentEx;
                }
                catch (Exception)
                {
                    //MessageBox.Show(
                    //  String.Format("Error: Creating COM component {0}." +
                    //    Environment.NewLine + "Detailed exception message: {1}",
                    //    ProgId,
                    //    ex.Message),
                    //  "Error",
                    //  MessageBoxButtons.OK,
                    //  MessageBoxIcon.Error);
                    isValid = false;
                    return;
                }

                if (MgaComponent == null)
                {
                    isValid = false;
                    return;
                }

                string _name = MgaComponent.ComponentParameter["name"] as string;
                string _description = MgaComponent.ComponentParameter["description"] as string;
                string _help = MgaComponent.ComponentParameter["help"] as string;
                if (_name != null)
                {
                    Name = _name;
                }
                if (_description != null)
                {
                    Description = _description;
                }
                if (_help != null)
                {
                    Help = _help;
                }
            }
            else
            {
                isValid = false;
            }
        }

        #endregion

        ~ComComponent()
        {
            Parameters = null;
            MgaComponent = null;
            GC.Collect();
            GC.WaitForPendingFinalizers();
            GC.Collect();
        }
    }
}
