/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.IO;
using System.Drawing;

using GME;
using GME.MGA;
using GME.MGA.Meta;
using CyPhyMasterInterpreter.Interpreter;
using GME.Util;
using System.Reflection;
using System.Drawing.Drawing2D;
using WorkflowDecorator;
using System.Windows.Forms;
using Newtonsoft.Json;
using META;
using System.Diagnostics;

namespace GME.CSharp
{
    [Guid("2108C2FF-AD6B-42DC-BDCC-C27957299303"),
    ProgId("MGA.Decorator.Workflow"),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class WorkflowDecorator : IMgaDecoratorCommon, IMgaElementDecorator
    {
        public WorkflowDecorator()
        {

        }
        public WorkflowDecorator(IMgaElementDecoratorEvents events)
        {

        }

        ~WorkflowDecorator()
        {
            interpreters = null;
            workflow = null;
            GC.Collect();
            GC.WaitForPendingFinalizers();
            GC.Collect();
        }

        // store the coordinates where it must be drawn
        int x, y;
        int w, h;

        // name
        string name = null;
        SizeF LabelSize;

        // color of place and its label
        Color color = Color.Black;
        Color labelColor = Color.Black;

        public void Destroy()
        {
        }

        public void DragEnter(out uint dropEffect, ulong pCOleDataObject, uint keyState, int pointx, int pointy, ulong transformHDC)
        {
            dropEffect = 0;
        }

        public void DragOver(out uint dropEffect, ulong pCOleDataObject, uint keyState, int pointx, int pointy, ulong transformHDC)
        {
            dropEffect = 0;
        }

        public void Draw(uint hdc)
        {
            // VERY important to cast to int
            System.IntPtr hdcptr;
            unchecked { hdcptr = (System.IntPtr)(int)hdc; }

            // Create graphics object
            Graphics g = Graphics.FromHdc(hdcptr);

            // Set up string format for the label
            StringFormat sf = new StringFormat(StringFormatFlags.NoClip);
            sf.Alignment = StringAlignment.Center;
            sf.LineAlignment = StringAlignment.Center;

            // Drawing style
            Pen pen = new Pen(color);
            Brush brush = new SolidBrush(color);
#if DEBUG
            // Draw the place
            g.DrawRectangle(pen, x, y, w, h);
#endif
            Color contentColor = Color.Black;
            string Content = "";
            //if (myobj == null)
            //{
            //  g.Dispose();
            //  return;
            //}
            if (LastMetaKind == "Task")
            {
                Icon icon = null;
                if (interpreters[TaskProgId].isValid)
                {
                    contentColor = Color.Blue;
                    Content = interpreters[TaskProgId].Name;
                    MgaRegistrar registrar = new MgaRegistrar();
                    string DllFileName = registrar.LocalDllPath[TaskProgId];
                    try
                    {
                        // TODO: we should read this from the registry...
                        // if the value is ,IDI_COMPICON get the icon from the dll
                        string iconFileNameGuess = Path.ChangeExtension(DllFileName, ".ico");

                        if (File.Exists(iconFileNameGuess))
                        {
                            icon = Icon.ExtractAssociatedIcon(iconFileNameGuess);
                        }
                        else
                        {
                            icon = Icon.ExtractAssociatedIcon(DllFileName);
                        }
                    }
                    catch (ArgumentException)
                    {
                    }
                }
                else
                {
                    contentColor = Color.Red;
                    Content = "INVALID" + Environment.NewLine + TaskProgId;
                    icon = InvalidTask;
                }
                using (Icon icon2 = new Icon(icon, IconWidth, IconHeight))
                    g.DrawIcon(icon2, new Rectangle(x, y, IconWidth, IconHeight));

                g.DrawString(
                    Content,
                    SystemFonts.DefaultFont,
                    new SolidBrush(contentColor),
                    new RectangleF(
                        x,
                        y + h + 15,
                        g.MeasureString(Content, SystemFonts.DefaultFont).Width,
                        g.MeasureString(Content, SystemFonts.DefaultFont).Height),
                    sf);

                /*
                g.DrawString(
                    Parameters,
                    SystemFonts.DefaultFont,
                    new SolidBrush(Color.DarkGreen),
                    new RectangleF(
                        x,
                        y + h + 25,
                        g.MeasureString(Parameters, SystemFonts.DefaultFont).Width,
                        g.MeasureString(Parameters, SystemFonts.DefaultFont).Height),
                    sf);
                 */
            }
            else if (LastMetaKind == "WorkflowRef")
            {
                int i = 0;
                Icon icon = null;
                foreach (ComComponent c in workflow)
                {
                    if (c.isValid)
                    {
                        MgaRegistrar registrar = new MgaRegistrar();
                        string DllFileName = registrar.LocalDllPath[c.ProgId];
                        try
                        {
                            // TODO: we should read this from the registry...
                            // if the value is ,IDI_COMPICON get the icon from the dll
                            string iconFileNameGuess = Path.ChangeExtension(DllFileName, ".ico");

                            if (File.Exists(iconFileNameGuess))
                            {
                                icon = Icon.ExtractAssociatedIcon(iconFileNameGuess);
                            }
                            else
                            {
                                icon = Icon.ExtractAssociatedIcon(DllFileName);
                            }
                        }
                        catch (ArgumentException)
                        {
                        }
                    }
                    else
                    {
                        // draw error image
                        icon = InvalidTask;
                    }
                    try
                    {
                        int IconStartX = x + (IconWidth + TaskPadding) * i;
                        g.DrawIcon(new Icon(icon, IconWidth, IconHeight),
                            IconStartX, y);

                        if (c != workflow.Reverse().FirstOrDefault())
                        {
                            Pen p = new Pen(Color.Black, LineWidth);
                            p.StartCap = LineStartCap;
                            p.EndCap = LineEndCap;
                            int LineStartX = IconStartX + IconWidth;
                            g.DrawLine(p,
                                new Point(
                                    LineStartX + LineStartPadding,
                                    y + IconHeight / 2),
                                new Point(
                                    LineStartX + TaskPadding - LineEndPadding,
                                    y + IconHeight / 2));
                        }
                        i++;
                    }
                    catch (ArgumentException)
                    {
                    }

                }
                if (workflow != null)
                {
                    if (workflow.Count == 0 ||
                        icon == null)
                    {
                        icon = UndefinedWorkFlow;
                        g.DrawIcon(
                            new Icon(icon, IconWidth, IconHeight),
                            x,
                            y);
                    }
                }
            }

            // Draw the label
            g.DrawString(name, SystemFonts.DefaultFont, new SolidBrush(labelColor), 
                new RectangleF(x + w / 2 - LabelSize.Width / 2, y + h + 5, LabelSize.Width, 10), sf);

            sf.Dispose();
            g.Dispose();
        }

        public void DrawEx(uint hdc, ulong gdip)
        {
            Draw(hdc);
        }

        public void SetLocation(int sx, int sy, int ex, int ey)
        {
            x = sx; y = sy; w = ex - sx; h = ey - sy;
        }

        public void Drop(ulong pCOleDataObject, uint dropEffect, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void DropFile(ulong hDropInfo, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void GetFeatures(out uint features)
        {
            features =
                Defines.F_MOUSEEVENTS;
            //Defines.F_HASLABEL |
            //Defines.F_RESIZABLE |
            //Defines.F_RESIZEAFTERMOD;
            //Defines.F_HASSTATE |
            //Defines.F_ANIMATION;
        }

        struct Defines
        {
            public const uint F_RESIZABLE = 1 << 0;
            public const uint F_MOUSEEVENTS = 1 << 1;
            public const uint F_HASLABEL = 1 << 2;
            public const uint F_HASSTATE = 1 << 3;
            public const uint F_HASPORTS = 1 << 4;
            public const uint F_ANIMATION = 1 << 5;
            public const uint F_IMGPATH = 1 << 6;
            public const uint F_RESIZEAFTERMOD = 1 << 7;
        };

        public void GetLabelLocation(out int sx, out int sy, out int ex, out int ey)
        {
            sx = x + w / 2 - (int)LabelSize.Width / 2;
            sy = y + h + 5;
            ex = sx + (int)LabelSize.Width;
            ey = y + w + 15;
        }

        public void GetLocation(
            out int sx,
            out int sy,
            out int ex,
            out int ey)
        {
            sx = x;
            sy = y;
            ex = x + w;
            ey = y + h;
        }

        public void GetMnemonic(out string mnemonic)
        {
            mnemonic = "MGA.Decorator.Workflow";
        }

        public void GetParam(string name, out object value)
        {
            value = null;
        }

        public void GetPortLocation(MgaFCO fco, out int sx, out int sy, out int ex, out int ey)
        {
            sx = sy = ex = ey = 0;
        }

        public MgaFCOs GetPorts()
        {
            return null;
        }

        public void GetPreferredSize(out int sizex, out int sizey)
        {
            sizex = w;
            sizey = h;
        }

        MgaFCO myobj = null;
        MgaMetaFCO mymetaobj = null;

        public void Initialize(
            MgaProject project,
            MgaMetaPart meta,
            MgaFCO obj)
        {
            // only store temporarily, they might be unavailable later
            myobj = obj;
            mymetaobj = null;
            LastMetaKind = myobj.Meta.Name;

            // obtain the metaobject
            GetMetaFCO(meta, out mymetaobj);

            if (obj != null)
            {
                // concrete object
                name = myobj.Name;
                if (myobj.Meta.Name == "Task")
                {
                    // task
                    // get progid check whether it is already in the cache
                    TaskProgId = myobj.StrAttrByName["COMName"];
                    if (interpreters.Keys.Contains(TaskProgId) == false)
                    {
                        // create an instance
                        ComComponent task = new ComComponent(TaskProgId);
                        interpreters.Add(TaskProgId, task);
                    }
                    // save parameters
                    Parameters = myobj.StrAttrByName["Parameters"];
                    h = IconHeight;
                    w = IconWidth;
                }
                else if (myobj.Meta.Name == "WorkflowRef")
                {
                    // Workflow reference get the tasks
                    // TODO: get those in the right order
                    workflow.Clear();
                    MgaReference wfRef = myobj as MgaReference;
                    Queue<string> items = new Queue<string>();
                    List<MgaAtom> tasks = new List<MgaAtom>();
                    List<MgaFCO> processed = new List<MgaFCO>();

                    if (wfRef.Referred != null)
                    {
                        tasks.AddRange(wfRef.Referred.ChildObjects.OfType<MgaAtom>());

                        MgaAtom StartTask = null;

                        StartTask = tasks.
                            Where(x => x.ExSrcFcos().Count() == 0).
                            FirstOrDefault();

                        if (StartTask != null)
                        {
                            MgaFCO NextTask = null;

                            workflow.Enqueue(
                                new ComComponent(StartTask.StrAttrByName["COMName"]));
                            processed.Add(StartTask as MgaFCO);

                            NextTask = StartTask.ExDstFcos().FirstOrDefault();

                            // avoid loops
                            while (NextTask != null &&
                                processed.Contains(NextTask) == false)
                            {
                                workflow.Enqueue(
                                    new ComComponent(NextTask.StrAttrByName["COMName"]));
                                NextTask = NextTask.ExDstFcos().FirstOrDefault();
                            }
                        }
                    }
                    h = IconHeight;
                    if (workflow.Count > 0)
                    {
                        w = IconWidth * workflow.Count +
                            TaskPadding * (workflow.Count - 1);
                    }
                    else
                    {
                        w = IconWidth;
                    }
                }
            }
            else
            {
                // not a concreter object (maybe in part browser?)
                name = mymetaobj.DisplayedName;
            }

            // to handle color and labelColor settings in GME
            if (!GetColorPreference(out color, "color"))
            {
                color = Color.Black;
            }
            if (!GetColorPreference(out labelColor, "nameColor"))
            {
                labelColor = Color.Black;
            }

            // null them for sure
            // myobj = null;
            mymetaobj = null;
        }

        public void InitializeEx(
            MgaProject project,
            MgaMetaPart meta,
            MgaFCO obj,
            IMgaCommonDecoratorEvents eventSink,
            ulong parentWnd)
        {
            try
            {
                Initialize(project, meta, obj);
                if (name != null)
                {
                    IntPtr parentHwnd;
                    unchecked { parentHwnd = (IntPtr)(int)parentWnd; }
                    using (Graphics g = Graphics.FromHwnd(parentHwnd))
                    {
                        LabelSize = g.MeasureString(name, SystemFonts.DefaultFont);
                    }
                }
            }
            catch (Exception ex)
            {
                // sometimes the mgafco is inaccessible

                Trace.TraceWarning(ex.ToString().Replace('\n','\t'));
            }
        }

        public void MenuItemSelected(uint menuItemId, uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        #region Mouse events

        public void MouseLeftButtonDoubleClick(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
            ComComponent c = new ComComponent(TaskProgId);
            if (c == null)
            {
                return;
            }
            Dictionary<string, string> ParametersDict = new Dictionary<string, string>();
            if (string.IsNullOrWhiteSpace(Parameters) == false)
            {
                try
                {
                    ParametersDict = (Dictionary<string, string>)JsonConvert.DeserializeObject(Parameters, typeof(Dictionary<string, string>));
                }
                catch (Newtonsoft.Json.JsonReaderException) { }
            }

            foreach (var parameter in c.ParameterList.Where(p => !ParametersDict.ContainsKey(p)))
            {
                ParametersDict[parameter] = c.GetParameterValue(parameter);
            }
            List<ParameterSettingsForm.Parameter> parameters;
            parameters = ParametersDict.Select(x => new ParameterSettingsForm.Parameter()
            {
                Name = x.Key,
                Value = x.Value,
            }).ToList();
            using (ParameterSettingsForm form = new ParameterSettingsForm(parameters))
            {
                if (c != null && c.isValid)
                {
                    form.ShowDialog();
                    Dictionary<String, String> d = form.parameters.ToDictionary(p => p.Name, p => p.Value);
                    string serialized = JsonConvert.SerializeObject(d, Formatting.Indented);
                    myobj.Project.BeginTransactionInNewTerr();
                    try
                    {
                        Parameters = myobj.StrAttrByName["Parameters"] = serialized;
                    }
                    catch
                    {
                        myobj.Project.AbortTransaction();
                    }
                    myobj.Project.CommitTransaction();
                }
            }
        }

        public void MouseLeftButtonDown(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseLeftButtonUp(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseMiddleButtonDoubleClick(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseMiddleButtonDown(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseMiddleButtonUp(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseMoved(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseRightButtonDoubleClick(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseRightButtonDown(ulong hCtxMenu, uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseRightButtonUp(uint nFlags, int pointx, int pointy, ulong transformHDC)
        {
        }

        public void MouseWheelTurned(uint nFlags, int distance, int pointx, int pointy, ulong transformHDC)
        {
        }

        #endregion

        public void OperationCanceled()
        {
        }

        public void SaveState()
        {
        }

        public void SetActive(bool isActive)
        {
        }

        public void SetParam(string name, object value)
        {
        }

        public void SetSelected(bool isSelected)
        {
        }

        /// <summary>
        /// Get the meta object
        /// </summary>
        /// <param name="metaPart">MetaPart</param>
        /// <param name="metaFco">returns the MetaFCO</param>
        /// <returns>true if succeeded</returns>
        bool GetMetaFCO(MgaMetaPart metaPart, out MgaMetaFCO metaFco)
        {
            metaFco = null;
            if (metaPart == null)
                return false;

            metaFco = metaPart.Role.Kind;
            return metaFco != null;
        }

        /// <summary>
        /// Return GME registry value
        /// </summary>
        /// <param name="val">returned value</param>
        /// <param name="path">path in registry</param>
        /// <returns>true if settins found and non-empty</returns>
        bool GetPreference(out string val, string path)
        {
            val = null;
            if (myobj != null)
            {
                val = myobj.RegistryValue[path];
            }
            else if (mymetaobj != null)
            {
                val = mymetaobj.RegistryValue[path];
            }
            return !string.IsNullOrEmpty(val);
        }

        // Get integer property from GME registry (if hex true, string is handled as hexa starting with 0x)
        bool GetPreference(out int val, string path, bool hex)
        {
            val = 0;
            string strVal;
            if (GetPreference(out strVal, path))
            {
                if (hex && strVal.Length >= 2)
                {
                    return int.TryParse(strVal.Substring(2), // omit 0x
                            System.Globalization.NumberStyles.HexNumber,
                            System.Globalization.NumberFormatInfo.InvariantInfo,
                            out val);
                }
                else if (!hex)
                {
                    return int.TryParse(strVal, out val);
                }
            }
            return false;
        }

        // Get color settings
        bool GetColorPreference(out Color color, string path)
        {
            int i;
            if (GetPreference(out i, path, true))
            {
                int r = (i & 0xff0000) >> 16;
                int g = (i & 0xff00) >> 8;
                int b = i & 0xff;
                color = Color.FromArgb(255, r, g, b);
                return true;
            }
            color = Color.Black;
            return false;
        }


        Dictionary<string, ComComponent> interpreters =
            new Dictionary<string, ComComponent>();

        Queue<ComComponent> workflow =
            new Queue<ComComponent>();

        public string TaskProgId { get; set; }

        public string Parameters { get; set; }

        public string LastMetaKind { get; set; }

        #region Tunable Parameters

        public int IconHeight { get { return 32; } }

        public int IconWidth { get { return 32; } }

        public int TaskPadding
        {
            get { return IconWidth; }
        }

        public Icon InvalidTask
        {
            get { return SystemIcons.Warning; }
        }

        public Icon UndefinedWorkFlow
        {
            get { return SystemIcons.Question; }
        }

        public int LineStartPadding { get { return 2; } }

        public int LineEndPadding { get { return 2; } }

        public LineCap LineStartCap { get { return LineCap.Flat; } }

        public LineCap LineEndCap { get { return LineCap.ArrowAnchor; } }

        public float LineWidth { get { return 1; } }

        #endregion



    }
}

