import java.io.File;
import java.net.URI;
import java.net.URL;
import java.net.URLDecoder;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.TreeMap;
import java.util.TreeSet;

import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;

import edu.vanderbilt.isis.cyphyml.FactoryRepository;
import edu.vanderbilt.isis.cyphyml.RootFolder;
import edu.vanderbilt.isis.cyphyml.RootFolderFileFactory;
import edu.vanderbilt.isis.cyphyml.TypeSpecifications;
import edu.vanderbilt.isis.cyphyml.Units;
import edu.vanderbilt.isis.cyphyml.conversion_based_unit;
import edu.vanderbilt.isis.cyphyml.derived_unit;
import edu.vanderbilt.isis.cyphyml.derived_unit_element;
import edu.vanderbilt.isis.cyphyml.named_unit;
import edu.vanderbilt.isis.cyphyml.reference_unit;
import edu.vanderbilt.isis.cyphyml.si_unit;
import edu.vanderbilt.isis.cyphyml.unit;


public class Qudt2CyPhyML {
	
	private static class RDFNodeComparator implements Comparator< RDFNode > {
		
		public int compare( RDFNode rdfNode1, RDFNode rdfNode2 ) {
			return rdfNode1.toString().compareTo( rdfNode2.toString() );
		}
		
		public boolean equals( Object obj ) {
			return obj instanceof RDFNodeComparator;
		}
		
	}

	private static class DimensionVector implements Cloneable {
		private TreeMap< RDFNode, Double > _vectorMagnitudes = new TreeMap<RDFNode, Double>( new RDFNodeComparator() );
		
		public void integrate( RDFNode rdfNode, double vectorMagnitude ) {
			if (  _vectorMagnitudes.containsKey( rdfNode )  ) {
				_vectorMagnitudes.put(  rdfNode, _vectorMagnitudes.get( rdfNode ) + vectorMagnitude  );
			} else {
				_vectorMagnitudes.put( rdfNode, vectorMagnitude );
			}
		}
		
		public DimensionVector add( DimensionVector other, double vectorMagnitude ) {
			DimensionVector dimensionVector = new DimensionVector();
			
			TreeSet< RDFNode > otherKeys = new TreeSet< RDFNode >( new RDFNodeComparator() );
			for( RDFNode rdfNode : other._vectorMagnitudes.keySet() ) {
				otherKeys.add( rdfNode );
			}
			
			for( RDFNode rdfNode : _vectorMagnitudes.keySet() ) {
				double value = _vectorMagnitudes.get( rdfNode );
				if (  otherKeys.contains( rdfNode )  ) {
					value += other._vectorMagnitudes.get( rdfNode ) * vectorMagnitude;
					otherKeys.remove( rdfNode );
				}
				dimensionVector._vectorMagnitudes.put( rdfNode, value );
			}
			for( RDFNode rdfNode : otherKeys ) {
				dimensionVector._vectorMagnitudes.put(  rdfNode, other._vectorMagnitudes.get( rdfNode ) * vectorMagnitude  );
			}
			
			return dimensionVector;
		}
		
		public DimensionVector clone() {
			DimensionVector dimensionVector = new DimensionVector();
			dimensionVector._vectorMagnitudes = ( TreeMap<RDFNode, Double> )_vectorMagnitudes.clone();
			return dimensionVector;
		}
		
		public TreeMap< RDFNode, Double > getVectorMagnitudes() {
			return _vectorMagnitudes;
		}
	}
	
	private Model _model = null;
	private HashMap< RDFNode, unit > _systemBaseUnitRDFNodeUnitMap = new HashMap< RDFNode, unit >();
	private HashMap< RDFNode, unit > _systemBaseQuantityKindRDFNodeUnitMap = new HashMap< RDFNode, unit >();
	private RootFolderFileFactory _rootFolderFileFactory = null;
	private Units _units = null;

	
	
	public Qudt2CyPhyML( String filename ) {

		URI qudtDirURI = null;
		try {
			_rootFolderFileFactory = FactoryRepository.getCyPhyMLRootFolderFileFactory();
			
			File file = new File( filename );
			RootFolder rootFolder = null;
			if ( file.exists() ) {
				rootFolder = _rootFolderFileFactory.open( file.getAbsolutePath() );
			} else {
				rootFolder = _rootFolderFileFactory.create( file.getAbsolutePath() );
				rootFolder.setname( "RootFolder" );
			}

			TypeSpecifications typeSpecifications = TypeSpecifications.create( rootFolder );
			_units = Units.create( typeSpecifications );

		
		    URL url = Qudt2CyPhyML.class.getResource( "Qudt2CyPhyML.class" );
		    qudtDirURI = new File(  URLDecoder.decode( url.getPath(), "UTF-8" )  ).getParentFile().toURI().resolve( "../qudt/" );
		    
		} catch ( Exception e ) {
			e.printStackTrace();
			System.exit( 1 );
		}

		_model = ModelFactory.createDefaultModel();
		_model.read(  qudtDirURI.resolve( "OSG_dimension-(v1.1).ttl" ).toString(), "TURTLE"  );
		_model.read(  qudtDirURI.resolve( "OSG_quantity-(v1.1).ttl" ).toString(), "TURTLE"  );
		_model.read(  qudtDirURI.resolve( "OSG_qudt-(v1.01).ttl" ).toString(), "TURTLE"  );
		_model.read(  qudtDirURI.resolve( "OVG_dimensionalunits-qudt-(v1.1).ttl" ).toString(), "TURTLE"  );
		_model.read(  qudtDirURI.resolve( "OVG_dimensions-qudt-(v1.1).ttl" ).toString(), "TURTLE"  );
		_model.read(  qudtDirURI.resolve( "OVG_quantities-qudt-(v1.1).ttl" ).toString(), "TURTLE"  );
		_model.read(  qudtDirURI.resolve( "OVG_units-qudt-(v1.1).ttl" ).toString(), "TURTLE"  );		
	}

	public void saveAndClose() {	
		try {
			_rootFolderFileFactory.save();
			//_rootFolderFileFactory.close();
		} catch( Exception e ) {
			e.printStackTrace();
		}
	}

	
	private void getBaseQuantityKindsAndUnits() {

		String systemBaseUnitQueryString =
		 "PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>\n" +
		 "PREFIX unit: <http://qudt.org/vocab/unit#>\n" +
		 "PREFIX qudt: <http://qudt.org/schema/qudt#>\n" +
		 "\n" +
		 "SELECT ?systemBaseUnit ?systemBaseQuantityKind ?name ?abbreviation ?symbol WHERE {\n" +
		 "  unit:SystemOfUnits_SI qudt:systemBaseUnit ?systemBaseUnit .\n" +
		 "  ?systemBaseUnit qudt:quantityKind ?systemBaseQuantityKind .\n" +
		 "  ?systemBaseUnit rdfs:label ?name .\n" +
		 "  OPTIONAL { ?systemBaseUnit qudt:abbreviation ?abbreviation } .\n" +
		 "  OPTIONAL { ?systemBaseUnit qudt:symbol ?symbol } .\n" +
		 "}";
		Query systemBaseUnitQuery = QueryFactory.create( systemBaseUnitQueryString );
		QueryExecution systemBaseUnitQueryExecution = QueryExecutionFactory.create( systemBaseUnitQuery, _model );
		try {

			ResultSet systemBaseUnitResultSet = systemBaseUnitQueryExecution.execSelect();
			while( systemBaseUnitResultSet.hasNext() ) {
				QuerySolution systemBaseUnitQuerySolution = systemBaseUnitResultSet.nextSolution();
				RDFNode systemBaseUnitRDFNode = systemBaseUnitQuerySolution.get( "systemBaseUnit" );

				RDFNode systemBaseQuantityKindRDFNode = systemBaseUnitQuerySolution.get( "systemBaseQuantityKind" );

				String systemBaseUnitName = systemBaseUnitQuerySolution.get( "name" ).asLiteral().getString();
				
				RDFNode systemBaseUnitAbbreviationRDFNode = systemBaseUnitQuerySolution.get( "abbreviation" );
				String systemBaseUnitAbbreviation = systemBaseUnitAbbreviationRDFNode == null ? "" : systemBaseUnitAbbreviationRDFNode.asLiteral().getString();
				
				RDFNode systemBaseUnitSymbolRDFNode = systemBaseUnitQuerySolution.get( "symbol" );
				String systemBaseUnitSymbol = systemBaseUnitSymbolRDFNode == null ? "" : systemBaseUnitSymbolRDFNode.asLiteral().getString();
				
				si_unit si_unit_var = si_unit.create( _units );
				si_unit_var.setname( systemBaseUnitName );
				si_unit_var.setAbbreviation( systemBaseUnitAbbreviation );
				si_unit_var.setSymbol( systemBaseUnitSymbol );
				si_unit_var.setFullName( systemBaseUnitName );
				
				if (  "Meter".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "metre" );
					si_unit_var.setlength_exponent( 1.0 );
				} else if (  "Kilogram".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "kilogram" );
					si_unit_var.setmass_exponent( 1.0 );
				} else if (  "Second".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "second" );
					si_unit_var.settime_exponent( 1.0 );
				} else if (  "Ampere".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "ampere" );
					si_unit_var.setelectric_current_exponent( 1.0 );
				} else if (  "Kelvin".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "kelvin" );
					si_unit_var.setthermodynamic_temperature_exponent( 1.0 );
				} else if (  "Mole".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "mole" );
					si_unit_var.setamount_of_substance_exponent( 1.0 );
				} else if (  "Candela".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "candela" );
					si_unit_var.setluminous_intensity_exponent( 1.0 );					
				} else if (  "Unitless".equals( systemBaseUnitName )  ) {
					si_unit_var.setsi_unit_name( "radian" );
				}
				
				_systemBaseUnitRDFNodeUnitMap.put( systemBaseUnitRDFNode, si_unit_var );
				_systemBaseQuantityKindRDFNodeUnitMap.put( systemBaseQuantityKindRDFNode, si_unit_var );
			}

		} catch( Exception e ) {
			e.printStackTrace();
			System.exit( 2 );
		} finally {
			systemBaseUnitQueryExecution.close();
		}
	}
	
	private DimensionVector getDerivedUnitDimensionsFromDimension( RDFNode dimensionRDFNode ) {

		DimensionVector dimensionVector = new DimensionVector();
		
		String dimensionVectorQueryString =
		 "PREFIX qudt:    <http://qudt.org/schema/qudt#>\n" +
		 "\n" +
		 "SELECT ?basisElement ?vectorMagnitude WHERE {\n" +
		 "  <" + dimensionRDFNode.asResource() + "> qudt:dimensionVector ?dimensionVector .\n" +
		 "  ?dimensionVector qudt:basisElement ?basisElement .\n" +
		 "  ?dimensionVector qudt:vectorMagnitude ?vectorMagnitude .\n" +
		 "}";
		
		Query dimensionVectorQuery = QueryFactory.create( dimensionVectorQueryString );
		QueryExecution dimensionVectorQueryExecution = QueryExecutionFactory.create( dimensionVectorQuery, _model );
		
		try {

			ResultSet dimensionVectorResultSet = dimensionVectorQueryExecution.execSelect();
			while( dimensionVectorResultSet.hasNext() ) {
				QuerySolution dimensionVectorQuerySolution = dimensionVectorResultSet.nextSolution();
				RDFNode basisElementRDFNode = dimensionVectorQuerySolution.get( "basisElement" );
				RDFNode vectorMagnitudeRDFNode = dimensionVectorQuerySolution.get( "vectorMagnitude" );
				double vectorMagnitude = vectorMagnitudeRDFNode.asLiteral().getDouble();

				if ( vectorMagnitude != 0 ) {
					if (  _systemBaseQuantityKindRDFNodeUnitMap.containsKey( basisElementRDFNode )  ) {
						dimensionVector.integrate( basisElementRDFNode, vectorMagnitude );
					} else {
						DimensionVector recursiveDimensionVector = getDerivedUnitDimensions( basisElementRDFNode );
						dimensionVector = dimensionVector.add( recursiveDimensionVector, vectorMagnitude );
					}
				}
			}
			
		} catch( Exception e ) {
			e.printStackTrace();
			System.exit( 2 );
		} finally {
			dimensionVectorQueryExecution.close();
		}

		return dimensionVector;
	}
	
	private DimensionVector getDerivedUnitDimensions( RDFNode quantityKindRDFNode ) {
		
		DimensionVector dimensionVector = new DimensionVector();
		
		String dimensionQueryString =
		 "PREFIX qudt:    <http://qudt.org/schema/qudt#>\n" +
		 "PREFIX qudt-quantity: <http://qudt.org/vocab/quantity#>\n" +
		 "\n" +
		 "SELECT ?dimension ?noReferenceQuantities WHERE { \n" +
		 "  qudt-quantity:SystemOfQuantities_SI qudt:systemDimension ?dimension .\n" +
		 "  ?dimension qudt:referenceQuantity <" + quantityKindRDFNode.asResource() + "> .\n" +
		 "  FILTER NOT EXISTS {\n" +
		 "    SELECT ?referenceQuantity WHERE {\n" +
		 "      ?dimension qudt:referenceQuantity ?referenceQuantity .\n" +
		 "      FILTER ( <" + quantityKindRDFNode.asResource() + "> != ?referenceQuantity )\n" +
		 "    }\n" +
		 "  }\n" +
		 "}";
		Query dimensionQuery = QueryFactory.create( dimensionQueryString );
		QueryExecution dimensionQueryExecution = QueryExecutionFactory.create( dimensionQuery, _model );
		
		try {
			ResultSet dimensionResultSet = dimensionQueryExecution.execSelect();
			
			if ( !dimensionResultSet.hasNext() ) {
				return dimensionVector;
			}

			QuerySolution dimensionQuerySolution = dimensionResultSet.next();
			RDFNode dimensionRDFNode = dimensionQuerySolution.get( "dimension" );

			dimensionVector = getDerivedUnitDimensionsFromDimension( dimensionRDFNode );

		} catch( Exception e ) {
			e.printStackTrace();
			System.exit( 2 );
		} finally {
			dimensionQueryExecution.close();
		}
		
		return dimensionVector;
	}

	private derived_unit getDerivedUnitRefs( RDFNode dimensionRDFNode ) {
		
		DimensionVector dimensionVector = getDerivedUnitDimensionsFromDimension( dimensionRDFNode );
		TreeMap< RDFNode, Double > vectorMagnitudes = dimensionVector.getVectorMagnitudes();
		
		if ( vectorMagnitudes.isEmpty() ) return null;
		
		derived_unit derived_unit_var = null;

		try {
			derived_unit_var = derived_unit.create( _units );
			for( Entry< RDFNode, Double > entry : vectorMagnitudes.entrySet() ) {
				derived_unit_element derived_unit_element_var = derived_unit_element.create( derived_unit_var );
				derived_unit_element_var.setexponent( entry.getValue() );
				named_unit named_unit_ref = (named_unit)_systemBaseQuantityKindRDFNodeUnitMap.get( entry.getKey() );
				derived_unit_element_var.setref(  named_unit_ref  );
				derived_unit_element_var.setname( named_unit_ref.getname() );
			}
		} catch ( Exception e ) {
			e.printStackTrace();
		}
		
		return derived_unit_var;
	}
	
	private void getUnits() {

		String quantityKindQueryString =
		 "PREFIX qudt:          <http://qudt.org/schema/qudt#>\n" +
		 "PREFIX qudt-quantity: <http://qudt.org/vocab/quantity#>\n" +
		 "\n" +
		 "SELECT ?dimension ?quantityKind WHERE {\n" +
		 "  qudt-quantity:SystemOfQuantities_SI qudt:systemDimension ?dimension .\n" +
		 "  ?dimension qudt:referenceQuantity ?quantityKind .\n" +
		 "  FILTER NOT EXISTS {\n" +
		 "    SELECT ?referenceQuantity WHERE {\n" +
		 "      ?dimension qudt:referenceQuantity ?referenceQuantity .\n" +
		 "      FILTER ( ?quantityKind != ?referenceQuantity ) .\n" +
		 "    }\n" +
		 "  }\n" +
		 "}";
		Query quantityKindQuery = QueryFactory.create( quantityKindQueryString );
		QueryExecution quantityKindQueryExecution = QueryExecutionFactory.create( quantityKindQuery, _model );
		
		try {
			
			ResultSet quantityKindResultSet = quantityKindQueryExecution.execSelect();
			while( quantityKindResultSet.hasNext() ) {

				QuerySolution quantityKindQuerySolution = quantityKindResultSet.nextSolution();
	
				RDFNode dimensionRDFNode = quantityKindQuerySolution.get( "dimension" );
				
				RDFNode quantityKindRDFNode = quantityKindQuerySolution.get( "quantityKind" );
				Resource quantityKindResource = quantityKindRDFNode.asResource();

				URI quantityKindURI = new URI( quantityKindResource.getURI() );
				String quantityKindName = quantityKindURI.getFragment();
				
				unit ref_unit_var = null;
				if (  _systemBaseQuantityKindRDFNodeUnitMap.containsKey( quantityKindRDFNode )  ) {
					ref_unit_var = _systemBaseQuantityKindRDFNodeUnitMap.get( quantityKindRDFNode );
				} else {
					ref_unit_var = getDerivedUnitRefs( dimensionRDFNode );
				}
				if ( ref_unit_var == null ) continue;

				String unitQueryString = 
				 "PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>\n" +
				 "PREFIX qudt:    <http://qudt.org/schema/qudt#>\n" +
				 "\n" +
				 "SELECT ?name ?conversionMultiplier ?conversionOffset ?abbreviation ?symbol WHERE {\n" +
				 "  ?unit qudt:quantityKind <" + quantityKindResource + "> .\n" +
				 "  ?unit rdfs:label ?name .\n" +
				 "  ?unit qudt:conversionMultiplier ?conversionMultiplier .\n" +
				 "  OPTIONAL { ?unit qudt:conversionOffset ?conversionOffset } .\n" +
				 "  OPTIONAL { ?unit qudt:abbreviation ?abbreviation } .\n" +
				 "  OPTIONAL { ?unit qudt:symbol ?symbol } .\n" +
				 "}";
				Query unitQuery = QueryFactory.create( unitQueryString );
				QueryExecution unitQueryExecution = QueryExecutionFactory.create( unitQuery, _model );

				HashSet< reference_unit > referenceUnitSet = new HashSet< reference_unit >();
				try {
					conversion_based_unit base_conversion_based_unit = null;
					ResultSet unitResultSet = unitQueryExecution.execSelect();
					while( unitResultSet.hasNext() ) {
						QuerySolution unitQuerySolution = unitResultSet.nextSolution();

						String unitName = unitQuerySolution.get( "name" ).asLiteral().getString();
						
						RDFNode unitConversionMultiplierRDFNode = unitQuerySolution.get( "conversionMultiplier" );
						double unitConversionMultiplier = unitConversionMultiplierRDFNode.asLiteral().getDouble();

						RDFNode unitConversionOffsetRDFNode = unitQuerySolution.get( "conversionOffset" );
						double unitConversionOffset = unitConversionOffsetRDFNode == null ? 0.0 : unitConversionOffsetRDFNode.asLiteral().getDouble();

						RDFNode unitAbbreviationRDFNode = unitQuerySolution.get( "abbreviation" );
						String unitAbbreviation = unitAbbreviationRDFNode == null ? "" : unitAbbreviationRDFNode.asLiteral().getString();
						
						RDFNode unitSymbolRDFNode = unitQuerySolution.get( "symbol" );
						String unitSymbol = unitSymbolRDFNode == null ? "" : unitSymbolRDFNode.asLiteral().getString();

						conversion_based_unit conversion_based_unit_var = conversion_based_unit.create( _units );
						conversion_based_unit_var.setname( unitName );
						conversion_based_unit_var.setFullName( unitName );
						conversion_based_unit_var.setAbbreviation( unitAbbreviation );
						conversion_based_unit_var.setSymbol( unitSymbol );
						
						if ( unitConversionMultiplier == 1.0 && unitConversionOffset == 0.0 ) {
							base_conversion_based_unit = conversion_based_unit_var;
						}
						
						reference_unit reference_unit_var = reference_unit.create( conversion_based_unit_var );
						reference_unit_var.setconversion_factor( unitConversionMultiplier );
						reference_unit_var.setconversion_offset( unitConversionOffset );
						referenceUnitSet.add( reference_unit_var );
					}
					
					if ( base_conversion_based_unit != null ) {
						if ( ! ( ref_unit_var instanceof si_unit )  ) {
							ref_unit_var.setname( base_conversion_based_unit.getname() );
							ref_unit_var.setFullName( base_conversion_based_unit.getFullName() );
							ref_unit_var.setAbbreviation( base_conversion_based_unit.getAbbreviation() );
							ref_unit_var.setSymbol( base_conversion_based_unit.getSymbol() );
						}
					} else {
						ref_unit_var.setname( quantityKindName );
					}
					
					Iterator< reference_unit > rusItr = referenceUnitSet.iterator();
					while ( rusItr.hasNext() ) {
						reference_unit reference_unit_var = rusItr.next();
						reference_unit_var.setref( ref_unit_var );
						reference_unit_var.setname( ref_unit_var.getname() );
					}
					
				} catch( Exception e ) {
					e.printStackTrace();
				} finally {
					unitQueryExecution.close();
				}

			}

		} catch( Exception e ) {
			e.printStackTrace();
			System.exit( 2 );
		} finally {
			quantityKindQueryExecution.close();
		}
		
	}
	
	/**
	 * @param args
	 */
	public static void main(String[] args) {

		String outputFilename = "CyPhyMLTypes.xml";
		if ( args.length == 1 ) {
			outputFilename = args[ 0 ];
		}
		Qudt2CyPhyML qudt2CyPhyML = new Qudt2CyPhyML( outputFilename );
		qudt2CyPhyML.getBaseQuantityKindsAndUnits();
		qudt2CyPhyML.getUnits();
		qudt2CyPhyML.saveAndClose();
		
	}

}