/* NAME:	Input file registry
 * AUTHOR:	Attila Vizhanyo - viza@isis.vanderbilt.edu
 *
 * Parser and registry for GR input file command line arguments 
 * of the format:
 *		FILE_ID="FILE_NAME"[;"COPY_NAME"][!"XSD_NAME"]
 * where the order of COPY_NAME and XSD_NAME is arbitrary.
 */
#ifndef INPUTFILEREGISTRY_H
#define INPUTFILEREGISTRY_H

#include <map>
#include <string>
#include <iostream>
#include <cctype> // for isspace()

// Input File Exception class
class InputFileEx
{
public:
	InputFileEx( const std::string& exMsg) { _exMsg= exMsg; std::cout << exMsg << std::endl;}
	const std::string& what() { return _exMsg;}
private:
	std::string _exMsg;
};

const std::string FILEID_SEPARATOR= "=";
const std::string COPYNAME_SEPARATOR= ";";
const std::string XSDNAME_SEPARATOR= "!";

#ifndef __UTIL_H
// Generic triplet class.
template<class T1, class T2, class T3> 
  class triplet 
  {
  public:
    ///    
    typedef T1 first_type;
    ///
    typedef T2 second_type;
    ///
    typedef T3 third_type;
    ///
    triplet()
      : first(first_type()), second(second_type()), third(third_type()) {}
    ///
    triplet(const first_type& f, const second_type& s, const third_type& t)
      : first(f), second(s), third(t) {}
    ///
    first_type first;
    ///
    second_type second;
    ///
    third_type third;
  };
  template<class T1, class T2, class T3> 
  inline triplet<T1, T2, T3> make_triplet(const T1& x, const T2& y, const T3& z)
  {
    return triplet<T1, T2, T3>(x, y, z); 
  }
#endif

// Keeps track of file names and copy names of file IDs.
class InputFileRegistry
{
public:
	// Parsess the expression of format: FILE_ID="FILE_NAME"[;"COPY_NAME"][!"XSD_NAME"]
	// and register FILE_NAME and COPY_NAME under the key FILE_ID.
	// Returns true, if 'fileExpr' is valid expression (registration successful).
	bool registerFile( const std::string& fileExpr)
	{
		std::string fileID, fileName, copyName, xsdName;
		if ( parseFileExpr( fileExpr, fileID, fileName, copyName, xsdName))
		{
			registerFile( fileID, fileName, copyName, xsdName);
			return true;
		}
		return false;
	}
	// Register fileName and copyName under the key fileID.
	void registerFile( const std::string& fileID, const std::string& fileName, const std::string& copyName, const std::string& xsdName)
	{
		_fileMap[ fileID]= make_triplet( fileName, copyName, xsdName);
	}
	// Returns file name belonging to previously registered 'fileID'. Throws InputFileEx, if fileID was not found.
	const std::string& getFileName( const std::string& fileID) const throw( InputFileEx)
	{
		FileMap_t::const_iterator findID= _fileMap.find( fileID);
		if( findID != _fileMap.end())
			return findID->second.first;
		throw InputFileEx("File ID: "+ fileID+ " not found.");
	}
	// Returns copy name belonging to previously registered 'fileID'. Throws InputFileEx, if fileID was not found.
	const std::string& getCopyName( const std::string& fileID) const throw ( InputFileEx)
	{
		FileMap_t::const_iterator findID= _fileMap.find( fileID);
		if( findID != _fileMap.end())
			return findID->second.second;
		throw InputFileEx("File ID: "+ fileID+ " not found.");
	}
	// Returns xsd name belonging to previously registered 'fileID'. Throws InputFileEx, if fileID was not found.
	const std::string& getXsdName( const std::string& fileID) const throw ( InputFileEx)
	{
		FileMap_t::const_iterator findID= _fileMap.find( fileID);
		if( findID != _fileMap.end())
			return findID->second.third;
		throw InputFileEx("File ID: "+ fileID+ " not found.");
	}
	// Parses 'fileExpr' of format: FILE_ID="FILE_NAME"[;"COPY_NAME"][!"XSD_NAME"]
	// Examples:	INPUT = "models\open this.mga" ; "..\write as this.xml"
	//				OUTPUT = "save.xml" ! "Transformations\XSDs\output.xsd"
	//				COPY = "open.xml" ; "copy.xml" ! "meta.xsd"
	// Returns true, if fileExpr is valid expression (parsing successful).
	bool parseFileExpr( const std::string& fileExpr, std::string& fileID, std::string& fileName, std::string& copyName, std::string& xsdName) const
	{
		bool isValid= false;
		std::string::size_type findEq= fileExpr.find( FILEID_SEPARATOR);
		if ( findEq != std::string::npos)
		{

#if _MSC_VER >= 1310 || !defined( _WIN32 )
			fileID.assign(fileExpr, 0, findEq);
			trim( fileID);
			std::string::size_type findCSep= fileExpr.find( COPYNAME_SEPARATOR);
			std::string::size_type findXSep= fileExpr.find( XSDNAME_SEPARATOR);
			if ( findCSep== std::string::npos)
			{
				if ( findXSep == std::string::npos)
					fileName.assign(fileExpr,findEq+ 1, fileExpr.length()-findEq-1);
				else
				{
					fileName.assign(fileExpr,findEq+ 1, findXSep- findEq- 1);
					xsdName.assign(fileExpr, findXSep+ 1, fileExpr.length()- findXSep-1);
				}
			}
			else 
			{
				if ( findXSep == std::string::npos)
				{
					fileName.assign( fileExpr,findEq+1, findCSep-findEq-1);
					copyName.assign( fileExpr,findCSep+1, fileExpr.length()-findCSep-1);
				}
				else
				{
					bool isCBeforeX= findCSep < findXSep;
					fileName.assign( fileExpr, findEq+1, ( isCBeforeX ? findCSep : findXSep) - findEq - 1);
					copyName.assign( fileExpr, findCSep+1, (isCBeforeX ?  findXSep : fileExpr.size()-1) -findCSep-1);
					xsdName.assign( fileExpr, findXSep+ 1, (isCBeforeX ?  fileExpr.size()-1 : findCSep) - findXSep -1 );
				}
			}
#else
			fileID.assign( fileExpr.begin(), findEq);
			trim( fileID);
			std::string::size_type findCSep= fileExpr.find( COPYNAME_SEPARATOR);
			std::string::size_type findXSep= fileExpr.find( XSDNAME_SEPARATOR);
			if ( findCSep== std::string::npos)
			{
				if ( findXSep == std::string::npos)
					fileName.assign( fileExpr.begin()+findEq+ 1, fileExpr.length()-findEq-1);
				else
				{
					fileName.assign( fileExpr.begin()+findEq+ 1, findXSep- findEq- 1);
					xsdName.assign( fileExpr.begin()+ findXSep+ 1, fileExpr.length()- findXSep-1);
				}
			}
			else 
			{
				if ( findXSep == std::string::npos)
				{
					fileName.assign( fileExpr.begin()+findEq+1, findCSep-findEq-1);
					copyName.assign( fileExpr.begin()+findCSep+1, fileExpr.length()-findCSep-1);
				}
				else
				{
					bool isCBeforeX= findCSep < findXSep;
					fileName.assign( fileExpr.begin()+findEq+1, ( isCBeforeX ? findCSep : findXSep) - findEq - 1);
					copyName.assign( fileExpr.begin()+findCSep+1, isCBeforeX ?  fileExpr.begin()+findXSep : fileExpr.end());
					xsdName.assign( fileExpr.begin()+ findXSep+ 1, isCBeforeX ?  fileExpr.end() : fileExpr.begin()+findCSep );
				}
			}
#endif

			trim( fileName);
			trim( copyName);
			trim( xsdName);
			cutApostrophes( fileName);
			cutApostrophes( copyName);
			cutApostrophes( xsdName);
			isValid= true;
		}
		return isValid;
	}

protected:
	// Trim leading and terminating white space characters.
	void trim( std::string& toTrim) const
	{
		if ( toTrim.empty())
			return;
		std::string::iterator i;
		for( i= toTrim.begin(); isspace( *i); ++i) {}
		toTrim.erase( toTrim.begin(), i);
		std::string::reverse_iterator ri;
		for( ri= toTrim.rbegin(); isspace( *ri); ++ri) {}
		toTrim.erase( ri.base(), toTrim.rbegin().base());
	}
	// Return false if toCut is not empty & the first and last characters are not ".
	bool cutApostrophes( std::string& toCut) const
	{
		if ( toCut.empty())
			return true;
		if ( ( toCut[ 0] == '\"') && ( toCut[ toCut.size()-1] == '\"'))
		{
			toCut.erase( 0, 1);
			toCut.erase( toCut.rbegin().base()-1, toCut.rbegin().base());
			return true;
		}
		return false;
	}

private:
	// File map keeps record of: fileID->( file name, copy name)
	typedef std::map< std::string, triplet< std::string, std::string, std::string > > FileMap_t;
	FileMap_t _fileMap;
};

class UseXSD
{
public:
	// Returns true if the file 'fname' uses dtd file to open/ create.
	bool operator()( const std::string& fname)
	{
		std::string ext( fname);
		std::string::size_type findExt= ext.rfind( '.');
		if ( findExt != std::string::npos)
			if( *(ext.begin()+ findExt+ 1) != '\\')
				ext.erase( ext.begin(), ext.begin()+ ++findExt);	// delete '.' too.
			else
				ext.clear();
		return ext == "xml";
	}
};

#endif	//INPUTFILEREGISTRY_H
